/*
 * Decompiled with CFR 0.152.
 */
package io.jenkins.plugins.swarmcloud.monitoring;

import com.github.dockerjava.api.model.ResourceRequirements;
import com.github.dockerjava.api.model.ResourceSpecs;
import com.github.dockerjava.api.model.Service;
import com.github.dockerjava.api.model.ServiceSpec;
import com.github.dockerjava.api.model.SwarmNode;
import com.github.dockerjava.api.model.SwarmNodeDescription;
import com.github.dockerjava.api.model.SwarmNodeResources;
import com.github.dockerjava.api.model.SwarmNodeState;
import com.github.dockerjava.api.model.SwarmNodeStatus;
import com.github.dockerjava.api.model.Task;
import com.github.dockerjava.api.model.TaskSpec;
import com.github.dockerjava.api.model.TaskState;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import hudson.Extension;
import hudson.model.AsyncPeriodicWork;
import hudson.model.TaskListener;
import hudson.slaves.Cloud;
import io.jenkins.plugins.swarmcloud.SwarmAgentTemplate;
import io.jenkins.plugins.swarmcloud.SwarmCloud;
import io.jenkins.plugins.swarmcloud.api.DockerSwarmClient;
import io.jenkins.plugins.swarmcloud.monitoring.ClusterStatus;
import io.jenkins.plugins.swarmcloud.monitoring.NodeInfo;
import io.jenkins.plugins.swarmcloud.monitoring.ServiceInfo;
import java.io.IOException;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import java.util.logging.Logger;
import jenkins.model.Jenkins;

@Extension
public class ClusterMonitor
extends AsyncPeriodicWork {
    private static final Logger LOGGER = Logger.getLogger(ClusterMonitor.class.getName());
    private static final long RECURRENCE_PERIOD = TimeUnit.SECONDS.toMillis(30L);
    private static final Map<String, ClusterStatus> statusCache = new ConcurrentHashMap<String, ClusterStatus>();
    private static volatile long lastUpdate = 0L;

    public ClusterMonitor() {
        super("Swarm Cluster Monitor");
    }

    public long getRecurrencePeriod() {
        return RECURRENCE_PERIOD;
    }

    @SuppressFBWarnings(value={"ST_WRITE_TO_STATIC_FROM_INSTANCE_METHOD"}, justification="Volatile static field intentionally used for global last update timestamp")
    protected void execute(TaskListener listener) throws IOException, InterruptedException {
        Jenkins jenkins = Jenkins.getInstanceOrNull();
        if (jenkins == null) {
            return;
        }
        for (Cloud cloud : jenkins.clouds) {
            if (!(cloud instanceof SwarmCloud)) continue;
            SwarmCloud swarmCloud = (SwarmCloud)cloud;
            try {
                ClusterStatus status = this.collectMetrics(swarmCloud);
                statusCache.put(swarmCloud.name, status);
            }
            catch (RuntimeException e) {
                LOGGER.log(Level.WARNING, "Failed to collect metrics for cloud: " + swarmCloud.name, e);
                statusCache.put(swarmCloud.name, ClusterStatus.error(swarmCloud.name, e.getMessage()));
            }
        }
        lastUpdate = System.currentTimeMillis();
    }

    @NonNull
    private ClusterStatus collectMetrics(SwarmCloud cloud) {
        ClusterStatus status = new ClusterStatus(cloud.name);
        try {
            DockerSwarmClient dockerClient = cloud.getDockerClient();
            status.setSwarmVersion(dockerClient.getSwarmVersion());
            List nodes = (List)dockerClient.getDockerClient().listSwarmNodesCmd().exec();
            status.setTotalNodes(nodes.size());
            int readyNodes = 0;
            int managerNodes = 0;
            long totalMemory = 0L;
            long totalCpu = 0L;
            for (SwarmNode node : nodes) {
                SwarmNodeDescription desc;
                SwarmNodeResources resources;
                SwarmNodeState nodeState;
                SwarmNodeStatus nodeStatus = node.getStatus();
                SwarmNodeState swarmNodeState = nodeState = nodeStatus != null ? nodeStatus.getState() : null;
                if (SwarmNodeState.READY.equals((Object)nodeState)) {
                    ++readyNodes;
                }
                if (node.getManagerStatus() != null) {
                    ++managerNodes;
                }
                SwarmNodeResources swarmNodeResources = resources = (desc = node.getDescription()) != null ? desc.getResources() : null;
                if (resources != null) {
                    Long memBytes = resources.getMemoryBytes();
                    Long cpuNano = resources.getNanoCPUs();
                    if (memBytes != null) {
                        totalMemory += memBytes.longValue();
                    }
                    if (cpuNano != null) {
                        totalCpu += cpuNano.longValue();
                    }
                }
                NodeInfo nodeInfo = new NodeInfo();
                nodeInfo.setId(node.getId());
                nodeInfo.setHostname(desc != null ? desc.getHostname() : "unknown");
                nodeInfo.setState(nodeState != null ? nodeState.name() : "unknown");
                if (resources != null) {
                    Long memBytes = resources.getMemoryBytes();
                    Long cpuNano = resources.getNanoCPUs();
                    nodeInfo.setMemoryBytes(memBytes != null ? memBytes : 0L);
                    nodeInfo.setCpuNanos(cpuNano != null ? cpuNano : 0L);
                }
                nodeInfo.setRole(node.getManagerStatus() != null ? "manager" : "worker");
                status.addNode(nodeInfo);
            }
            status.setReadyNodes(readyNodes);
            status.setManagerNodes(managerNodes);
            status.setTotalMemory(totalMemory);
            status.setTotalCpu((double)totalCpu / 1.0E9);
            List<Service> services = dockerClient.listJenkinsServices();
            status.setActiveServices(services.size());
            int runningServices = 0;
            int pendingServices = 0;
            int failedServices = 0;
            long reservedMemory = 0L;
            long reservedCpuNano = 0L;
            for (Service service : services) {
                Date createdAt;
                Map labels;
                String serviceId = service.getId();
                if (serviceId == null) continue;
                ServiceInfo info = new ServiceInfo();
                info.setId(serviceId);
                ServiceSpec serviceSpec = service.getSpec();
                info.setName(serviceSpec != null ? serviceSpec.getName() : "unknown");
                if (serviceSpec != null && (labels = serviceSpec.getLabels()) != null) {
                    info.setTemplateName((String)labels.get("jenkins.template"));
                }
                if ((createdAt = service.getCreatedAt()) != null) {
                    info.setCreatedTime(createdAt.getTime());
                }
                List<Task> tasks = dockerClient.getServiceTasks(serviceId);
                boolean hasRunning = false;
                boolean hasPending = false;
                boolean hasFailed = false;
                boolean hasComplete = false;
                boolean hasShutdown = false;
                String lastError = null;
                for (Task task : tasks) {
                    if (task.getStatus() == null) continue;
                    TaskState state = task.getStatus().getState();
                    if (state == TaskState.RUNNING) {
                        hasRunning = true;
                        reservedMemory += this.getTaskReservedMemory(task);
                        reservedCpuNano += this.getTaskReservedCpu(task);
                        continue;
                    }
                    if (state == TaskState.PENDING || state == TaskState.ASSIGNED || state == TaskState.ACCEPTED || state == TaskState.PREPARING || state == TaskState.READY || state == TaskState.STARTING) {
                        hasPending = true;
                        continue;
                    }
                    if (state == TaskState.COMPLETE) {
                        hasComplete = true;
                        continue;
                    }
                    if (state == TaskState.SHUTDOWN || state == TaskState.ORPHANED) {
                        hasShutdown = true;
                        continue;
                    }
                    if (state != TaskState.FAILED && state != TaskState.REJECTED) continue;
                    hasFailed = true;
                    if (task.getStatus().getErr() == null) continue;
                    lastError = task.getStatus().getErr();
                }
                if (hasRunning) {
                    info.setState("running");
                    ++runningServices;
                } else if (hasPending) {
                    info.setState("pending");
                    ++pendingServices;
                    if (hasFailed && lastError != null) {
                        info.setError(lastError);
                    }
                } else if (hasComplete) {
                    info.setState("complete");
                } else if (hasShutdown) {
                    info.setState("shutdown");
                } else if (hasFailed) {
                    info.setState("failed");
                    ++failedServices;
                    if (lastError != null) {
                        info.setError(lastError);
                    }
                } else if (tasks.isEmpty()) {
                    info.setState("stopped");
                } else {
                    info.setState("unknown");
                }
                status.addService(info);
            }
            status.setRunningTasks(runningServices);
            status.setPendingTasks(pendingServices);
            status.setFailedTasks(failedServices);
            status.setReservedMemory(reservedMemory);
            status.setReservedCpu((double)reservedCpuNano / 1.0E9);
            status.setUsedMemory(reservedMemory);
            status.setUsedCpu((double)reservedCpuNano / 1.0E9);
            status.setMaxAgents(cloud.getMaxConcurrentAgents());
            status.setCurrentAgents(cloud.countCurrentAgents());
            status.setTemplateCount(cloud.getTemplates().size());
            status.setHealthy(true);
            status.setLastUpdate(System.currentTimeMillis());
            this.synchronizeTemplateCounters(cloud, services);
        }
        catch (RuntimeException e) {
            LOGGER.log(Level.SEVERE, "Error collecting metrics for cloud: " + cloud.name, e);
            status.setHealthy(false);
            status.setErrorMessage(e.getMessage());
        }
        return status;
    }

    @NonNull
    public static ClusterStatus getStatus(@NonNull String cloudName) {
        ClusterStatus status = statusCache.get(cloudName);
        return status != null ? status : ClusterStatus.unknown(cloudName);
    }

    @NonNull
    public static Map<String, ClusterStatus> getAllStatuses() {
        return Map.copyOf(statusCache);
    }

    public static long getLastUpdate() {
        return lastUpdate;
    }

    public static void refreshNow(@NonNull String cloudName) {
        Jenkins jenkins = Jenkins.getInstanceOrNull();
        if (jenkins == null) {
            return;
        }
        for (Cloud cloud : jenkins.clouds) {
            if (!(cloud instanceof SwarmCloud) || !cloud.name.equals(cloudName)) continue;
            ClusterMonitor monitor = jenkins.getExtensionList(ClusterMonitor.class).stream().findFirst().orElse(null);
            if (monitor == null) break;
            try {
                ClusterStatus status = monitor.collectMetrics((SwarmCloud)cloud);
                statusCache.put(cloudName, status);
            }
            catch (RuntimeException e) {
                LOGGER.log(Level.WARNING, "Refresh failed: " + cloudName, e);
            }
            break;
        }
    }

    private void synchronizeTemplateCounters(SwarmCloud cloud, List<Service> services) {
        HashMap<String, Integer> templateServiceCount = new HashMap<String, Integer>();
        for (Service service : services) {
            String templateName;
            Map labels;
            ServiceSpec serviceSpec = service.getSpec();
            if (serviceSpec == null || (labels = serviceSpec.getLabels()) == null || (templateName = (String)labels.get("jenkins.template")) == null) continue;
            templateServiceCount.merge(templateName, 1, Integer::sum);
        }
        for (SwarmAgentTemplate template : cloud.getTemplates()) {
            String templateName = template.getName();
            int actualCount = templateServiceCount.getOrDefault(templateName, 0);
            AtomicInteger counter = template.getCurrentInstancesCounter();
            int currentCount = counter.get();
            if (currentCount == actualCount || currentCount <= actualCount && actualCount - currentCount <= 1 || !counter.compareAndSet(currentCount, actualCount)) continue;
            LOGGER.log(Level.INFO, "Synchronized template ''{0}'' counter: {1} -> {2}", new Object[]{templateName, currentCount, actualCount});
        }
    }

    private long getTaskReservedMemory(Task task) {
        Long memBytes;
        TaskSpec taskSpec = task.getSpec();
        if (taskSpec == null) {
            return 0L;
        }
        ResourceRequirements resources = taskSpec.getResources();
        if (resources == null) {
            return 0L;
        }
        ResourceSpecs reservations = resources.getReservations();
        Long l = memBytes = reservations != null ? reservations.getMemoryBytes() : null;
        if (memBytes != null) {
            return memBytes;
        }
        ResourceSpecs limits = resources.getLimits();
        Long l2 = memBytes = limits != null ? limits.getMemoryBytes() : null;
        if (memBytes != null) {
            return memBytes;
        }
        return 0L;
    }

    private long getTaskReservedCpu(Task task) {
        Long cpuNano;
        TaskSpec taskSpec = task.getSpec();
        if (taskSpec == null) {
            return 0L;
        }
        ResourceRequirements resources = taskSpec.getResources();
        if (resources == null) {
            return 0L;
        }
        ResourceSpecs reservations = resources.getReservations();
        Long l = cpuNano = reservations != null ? reservations.getNanoCPUs() : null;
        if (cpuNano != null) {
            return cpuNano;
        }
        ResourceSpecs limits = resources.getLimits();
        Long l2 = cpuNano = limits != null ? limits.getNanoCPUs() : null;
        if (cpuNano != null) {
            return cpuNano;
        }
        return 0L;
    }
}

