/*
 * Decompiled with CFR 0.152.
 */
package land.oras.utils;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermission;
import java.util.EnumSet;
import java.util.Set;
import java.util.stream.Stream;
import land.oras.LocalPath;
import land.oras.exception.OrasException;
import land.oras.utils.SupportedCompression;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.compress.archivers.tar.TarArchiveOutputStream;
import org.apache.commons.compress.compressors.gzip.GzipCompressorInputStream;
import org.apache.commons.compress.compressors.gzip.GzipCompressorOutputStream;
import org.apache.commons.compress.compressors.zstandard.ZstdCompressorInputStream;
import org.apache.commons.compress.compressors.zstandard.ZstdCompressorOutputStream;
import org.jspecify.annotations.NullMarked;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NullMarked
public final class ArchiveUtils {
    private static final Logger LOG = LoggerFactory.getLogger(ArchiveUtils.class);

    private ArchiveUtils() {
    }

    public static Path createTempTar() {
        try {
            return Files.createTempFile("oras", ".tar", new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new OrasException("Failed to create temporary archive", e);
        }
    }

    public static Path createTempDir() {
        try {
            return Files.createTempDirectory("oras", new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new OrasException("Failed to create temporary directory", e);
        }
    }

    public static LocalPath tar(LocalPath sourceDir) {
        Path tarFile = ArchiveUtils.createTempTar();
        boolean isAbsolute = sourceDir.getPath().isAbsolute();
        try (OutputStream fos = Files.newOutputStream(tarFile, new OpenOption[0]);
             BufferedOutputStream bos = new BufferedOutputStream(fos);
             TarArchiveOutputStream taos = new TarArchiveOutputStream((OutputStream)bos);){
            taos.setLongFileMode(3);
            try (Stream<Path> paths = Files.walk(sourceDir.getPath(), new FileVisitOption[0]);){
                paths.forEach(path -> {
                    LOG.trace("Visiting path: {}", path);
                    try {
                        Path baseName = isAbsolute ? sourceDir.getPath().getFileName() : sourceDir.getPath();
                        Path relativePath = baseName.resolve(sourceDir.getPath().relativize((Path)path));
                        if (relativePath.toString().isEmpty()) {
                            LOG.trace("Skipping root directory: {}", path);
                            return;
                        }
                        String entryName = relativePath.toString();
                        TarArchiveEntry entry = null;
                        BasicFileAttributes attrs = Files.readAttributes(path, BasicFileAttributes.class, new LinkOption[0]);
                        if (Files.isSymbolicLink(path)) {
                            LOG.trace("Adding symlink entry: {}", (Object)entryName);
                            Path linkTarget = Files.readSymbolicLink(path);
                            entry = new TarArchiveEntry(entryName, 50);
                            entry.setLinkName(linkTarget.toString());
                            entry.setSize(0L);
                        } else {
                            LOG.trace("Adding entry: {}", (Object)entryName);
                            entry = new TarArchiveEntry(path.toFile(), entryName);
                            entry.setSize(attrs.isRegularFile() ? Files.size(path) : 0L);
                        }
                        Set<PosixFilePermission> permissions = Files.getPosixFilePermissions(path, new LinkOption[0]);
                        int mode = ArchiveUtils.permissionsToMode(permissions);
                        LOG.trace("Permissions: {}", permissions);
                        LOG.trace("Mode: {}", (Object)mode);
                        entry.setUserId(0);
                        entry.setGroupId(0);
                        entry.setUserName("");
                        entry.setGroupName("");
                        entry.setMode(mode);
                        taos.putArchiveEntry(entry);
                        if (attrs.isRegularFile() && !entry.isSymbolicLink()) {
                            try (InputStream fis = Files.newInputStream(path, new OpenOption[0]);){
                                fis.transferTo((OutputStream)taos);
                            }
                        }
                        taos.closeArchiveEntry();
                    }
                    catch (IOException e) {
                        throw new OrasException("Failed to create tar.gz file", e);
                    }
                });
            }
            catch (IOException e) {
                throw new OrasException("Failed to create tar.gz file", e);
            }
        }
        catch (IOException e) {
            throw new OrasException("Failed to create tar.gz file", e);
        }
        return LocalPath.of(tarFile, "application/vnd.oci.image.layer.v1.tar");
    }

    public static LocalPath tarcompress(LocalPath sourceDir, String mediaType) {
        return ArchiveUtils.compress(ArchiveUtils.tar(sourceDir), mediaType);
    }

    static void ensureSafeEntry(TarArchiveEntry entry, Path target) throws IOException {
        Path normalizedTarget;
        Path outputPath = target.resolve(entry.getName()).normalize();
        if (!outputPath.startsWith(normalizedTarget = target.toAbsolutePath().normalize())) {
            throw new IOException("Entry is outside of the target dir: " + String.valueOf(target));
        }
    }

    public static void untar(Path path, Path target) {
        try {
            ArchiveUtils.untar(Files.newInputStream(path, new OpenOption[0]), target);
        }
        catch (IOException e) {
            throw new OrasException("Failed to extract tar.gz file", e);
        }
    }

    public static void uncompressuntar(Path path, Path target, String mediaType) {
        try {
            LocalPath tar = ArchiveUtils.uncompress(Files.newInputStream(path, new OpenOption[0]), mediaType);
            ArchiveUtils.untar(tar.getPath(), target);
        }
        catch (IOException e) {
            throw new OrasException("Failed to extract tar.gz file", e);
        }
    }

    public static Path uncompressuntar(Path path, String mediaType) {
        Path tempDir = ArchiveUtils.createTempDir();
        ArchiveUtils.uncompressuntar(path, tempDir, mediaType);
        return tempDir;
    }

    public static Path untar(Path path) {
        Path tempDir = ArchiveUtils.createTempDir();
        ArchiveUtils.untar(path, tempDir);
        return tempDir;
    }

    public static void untar(InputStream fis, Path target) {
        try (BufferedInputStream bis = new BufferedInputStream(fis);
             TarArchiveInputStream tais = new TarArchiveInputStream((InputStream)bis);){
            TarArchiveEntry entry;
            while ((entry = tais.getNextEntry()) != null) {
                Path outputPath = target.resolve(entry.getName()).normalize();
                ArchiveUtils.ensureSafeEntry(entry, target);
                LOG.trace("Extracting entry: {}", (Object)entry.getName());
                if (entry.isDirectory()) {
                    LOG.debug("Extracting directory: {}", (Object)entry.getName());
                    Files.createDirectories(outputPath, new FileAttribute[0]);
                    continue;
                }
                LOG.trace("Creating directories for file: {}", (Object)outputPath.getParent());
                Files.createDirectories(outputPath.getParent(), new FileAttribute[0]);
                if (entry.isSymbolicLink()) {
                    LOG.trace("Extracting symlink {} to: {}", (Object)outputPath, (Object)entry.getLinkName());
                    Files.createSymbolicLink(outputPath, Paths.get(entry.getLinkName(), new String[0]), new FileAttribute[0]);
                    continue;
                }
                try (OutputStream out = Files.newOutputStream(outputPath, new OpenOption[0]);){
                    tais.transferTo(out);
                }
                Files.setPosixFilePermissions(outputPath, ArchiveUtils.convertToPosixPermissions(entry.getMode()));
            }
        }
        catch (IOException e) {
            throw new OrasException("Failed to extract tar.gz file", e);
        }
    }

    public static LocalPath compress(LocalPath path, String mediaType) {
        return SupportedCompression.fromMediaType(mediaType).compress(path);
    }

    public static LocalPath uncompress(InputStream is, String mediaType) {
        return SupportedCompression.fromMediaType(mediaType).uncompress(is);
    }

    static LocalPath compressZstd(LocalPath tarFile) {
        LOG.trace("Compressing tar file to zstd archive");
        Path tarGzFile = Paths.get(tarFile.toString() + ".gz", new String[0]);
        try (InputStream fis = Files.newInputStream(tarFile.getPath(), new OpenOption[0]);
             BufferedInputStream bis = new BufferedInputStream(fis);
             OutputStream fos = Files.newOutputStream(tarGzFile, new OpenOption[0]);
             BufferedOutputStream bos = new BufferedOutputStream(fos);
             ZstdCompressorOutputStream zstdos = new ZstdCompressorOutputStream((OutputStream)bos);){
            bis.transferTo((OutputStream)zstdos);
        }
        catch (IOException e) {
            throw new OrasException("Failed to compress tar file to zstd archive", e);
        }
        return LocalPath.of(tarGzFile, "application/vnd.oci.image.layer.v1.tar+zstd");
    }

    static LocalPath compressGzip(LocalPath tarFile) {
        LOG.trace("Compressing tar file to gz archive");
        Path tarGzFile = Paths.get(tarFile.toString() + ".gz", new String[0]);
        try (InputStream fis = Files.newInputStream(tarFile.getPath(), new OpenOption[0]);
             BufferedInputStream bis = new BufferedInputStream(fis);
             OutputStream fos = Files.newOutputStream(tarGzFile, new OpenOption[0]);
             BufferedOutputStream bos = new BufferedOutputStream(fos);
             GzipCompressorOutputStream gzos = new GzipCompressorOutputStream((OutputStream)bos);){
            bis.transferTo((OutputStream)gzos);
        }
        catch (IOException e) {
            throw new OrasException("Failed to compress tar file to gz archive", e);
        }
        return LocalPath.of(tarGzFile, "application/vnd.oci.image.layer.v1.tar+gzip");
    }

    static LocalPath uncompressGzip(InputStream inputStream) {
        LOG.trace("Uncompressing tar.gz file");
        Path tarFile = ArchiveUtils.createTempTar();
        try (BufferedInputStream bis = new BufferedInputStream(inputStream);
             GzipCompressorInputStream gzis = new GzipCompressorInputStream((InputStream)bis);
             OutputStream fos = Files.newOutputStream(tarFile, new OpenOption[0]);
             BufferedOutputStream bos = new BufferedOutputStream(fos);){
            gzis.transferTo((OutputStream)bos);
        }
        catch (IOException e) {
            throw new OrasException("Failed to uncompress tar.gz file", e);
        }
        return LocalPath.of(tarFile, "application/vnd.oci.image.layer.v1.tar");
    }

    static LocalPath uncompressZstd(InputStream inputStream) {
        LOG.trace("Uncompressing zstd file");
        Path tarFile = ArchiveUtils.createTempTar();
        try (BufferedInputStream bis = new BufferedInputStream(inputStream);
             ZstdCompressorInputStream gzis = new ZstdCompressorInputStream((InputStream)bis);
             OutputStream fos = Files.newOutputStream(tarFile, new OpenOption[0]);
             BufferedOutputStream bos = new BufferedOutputStream(fos);){
            gzis.transferTo((OutputStream)bos);
        }
        catch (IOException e) {
            throw new OrasException("Failed to uncompress tar.zstd file", e);
        }
        return LocalPath.of(tarFile, "application/vnd.oci.image.layer.v1.tar");
    }

    private static int permissionsToMode(Set<PosixFilePermission> permissions) {
        int mode = 0;
        if (permissions.contains((Object)PosixFilePermission.OWNER_READ)) {
            mode |= 0x100;
        }
        if (permissions.contains((Object)PosixFilePermission.OWNER_WRITE)) {
            mode |= 0x80;
        }
        if (permissions.contains((Object)PosixFilePermission.OWNER_EXECUTE)) {
            mode |= 0x40;
        }
        if (permissions.contains((Object)PosixFilePermission.GROUP_READ)) {
            mode |= 0x20;
        }
        if (permissions.contains((Object)PosixFilePermission.GROUP_WRITE)) {
            mode |= 0x10;
        }
        if (permissions.contains((Object)PosixFilePermission.GROUP_EXECUTE)) {
            mode |= 8;
        }
        if (permissions.contains((Object)PosixFilePermission.OTHERS_READ)) {
            mode |= 4;
        }
        if (permissions.contains((Object)PosixFilePermission.OTHERS_WRITE)) {
            mode |= 2;
        }
        if (permissions.contains((Object)PosixFilePermission.OTHERS_EXECUTE)) {
            mode |= 1;
        }
        return mode;
    }

    private static Set<PosixFilePermission> convertToPosixPermissions(int mode) {
        EnumSet<PosixFilePermission> permissions = EnumSet.noneOf(PosixFilePermission.class);
        if ((mode & 0x100) != 0) {
            permissions.add(PosixFilePermission.OWNER_READ);
        }
        if ((mode & 0x80) != 0) {
            permissions.add(PosixFilePermission.OWNER_WRITE);
        }
        if ((mode & 0x40) != 0) {
            permissions.add(PosixFilePermission.OWNER_EXECUTE);
        }
        if ((mode & 0x20) != 0) {
            permissions.add(PosixFilePermission.GROUP_READ);
        }
        if ((mode & 0x10) != 0) {
            permissions.add(PosixFilePermission.GROUP_WRITE);
        }
        if ((mode & 8) != 0) {
            permissions.add(PosixFilePermission.GROUP_EXECUTE);
        }
        if ((mode & 4) != 0) {
            permissions.add(PosixFilePermission.OTHERS_READ);
        }
        if ((mode & 2) != 0) {
            permissions.add(PosixFilePermission.OTHERS_WRITE);
        }
        if ((mode & 1) != 0) {
            permissions.add(PosixFilePermission.OTHERS_EXECUTE);
        }
        return permissions;
    }
}

