/*
 * Decompiled with CFR 0.152.
 */
package land.oras;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import land.oras.Annotations;
import land.oras.ArtifactType;
import land.oras.Config;
import land.oras.ContainerRef;
import land.oras.Descriptor;
import land.oras.Index;
import land.oras.Layer;
import land.oras.LocalPath;
import land.oras.Manifest;
import land.oras.ManifestDescriptor;
import land.oras.OCI;
import land.oras.Referrers;
import land.oras.Repositories;
import land.oras.Tags;
import land.oras.auth.AuthProvider;
import land.oras.auth.AuthStoreAuthenticationProvider;
import land.oras.auth.HttpClient;
import land.oras.auth.NoAuthProvider;
import land.oras.auth.Scope;
import land.oras.auth.Scopes;
import land.oras.auth.UsernamePasswordProvider;
import land.oras.exception.OrasException;
import land.oras.utils.ArchiveUtils;
import land.oras.utils.Const;
import land.oras.utils.JsonUtils;
import land.oras.utils.SupportedAlgorithm;
import org.jspecify.annotations.NullMarked;
import org.jspecify.annotations.Nullable;

@NullMarked
public final class Registry
extends OCI<ContainerRef> {
    private HttpClient client;
    private AuthProvider authProvider = new NoAuthProvider();
    private boolean insecure;
    private @Nullable String registry;
    private boolean skipTlsVerify;

    private Registry() {
        this.client = HttpClient.Builder.builder().build();
    }

    public static Builder builder() {
        return Builder.builder();
    }

    private void setInsecure(boolean insecure) {
        this.insecure = insecure;
    }

    private void setAuthProvider(AuthProvider authProvider) {
        this.authProvider = authProvider;
    }

    private void setRegistry(String registry) {
        this.registry = registry;
    }

    private void setSkipTlsVerify(boolean skipTlsVerify) {
        this.skipTlsVerify = skipTlsVerify;
    }

    private Registry build() {
        this.client = HttpClient.Builder.builder().withSkipTlsVerify(this.skipTlsVerify).build();
        return this;
    }

    public String getScheme() {
        return this.insecure ? "http" : "https";
    }

    public @Nullable String getRegistry() {
        return this.registry;
    }

    @Override
    public Tags getTags(ContainerRef containerRef) {
        URI uri = URI.create("%s://%s".formatted(this.getScheme(), containerRef.forRegistry(this).getTagsPath(this)));
        HttpClient.ResponseWrapper<String> response = this.client.get(uri, Map.of("Accept", "application/json"), Scopes.of(this, containerRef, new Scope[0]), this.authProvider);
        this.handleError(response);
        return JsonUtils.fromJson(response.response(), Tags.class);
    }

    @Override
    public Repositories getRepositories() {
        ContainerRef containerRef = ContainerRef.parse("default").forRegistry(this);
        URI uri = URI.create("%s://%s".formatted(this.getScheme(), containerRef.getRepositoriesPath(this)));
        HttpClient.ResponseWrapper<String> response = this.client.get(uri, Map.of("Accept", "application/json"), Scopes.of(this, containerRef, new Scope[0]), this.authProvider);
        this.handleError(response);
        return JsonUtils.fromJson(response.response(), Repositories.class);
    }

    @Override
    public Referrers getReferrers(ContainerRef containerRef, @Nullable ArtifactType artifactType) {
        if (containerRef.getDigest() == null) {
            throw new OrasException("Digest is required to get referrers");
        }
        URI uri = URI.create("%s://%s".formatted(this.getScheme(), containerRef.forRegistry(this).getReferrersPath(this, artifactType)));
        HttpClient.ResponseWrapper<String> response = this.client.get(uri, Map.of("Accept", "application/vnd.oci.image.index.v1+json"), Scopes.of(this, containerRef, new Scope[0]), this.authProvider);
        this.handleError(response);
        return JsonUtils.fromJson(response.response(), Referrers.class);
    }

    public void deleteManifest(ContainerRef containerRef) {
        URI uri = URI.create("%s://%s".formatted(this.getScheme(), containerRef.forRegistry(this).getManifestsPath(this)));
        HttpClient.ResponseWrapper<String> response = this.client.delete(uri, Map.of(), Scopes.of(this, containerRef, new Scope[0]), this.authProvider);
        this.logResponse(response);
        this.handleError(response);
    }

    @Override
    public Manifest pushManifest(ContainerRef containerRef, Manifest manifest) {
        Map<String, String> annotations = manifest.getAnnotations();
        if (!annotations.containsKey("org.opencontainers.image.created") && containerRef.getDigest() == null) {
            HashMap<String, String> manifestAnnotations = new HashMap<String, String>(annotations);
            manifestAnnotations.put("org.opencontainers.image.created", Const.currentTimestamp());
            manifest = manifest.withAnnotations(manifestAnnotations);
        }
        URI uri = URI.create("%s://%s".formatted(this.getScheme(), containerRef.forRegistry(this).getManifestsPath(this)));
        byte[] manifestData = manifest.getJson() != null ? manifest.getJson().getBytes() : manifest.toJson().getBytes();
        LOG.debug("Manifest data to push: {}", (Object)new String(manifestData, StandardCharsets.UTF_8));
        HttpClient.ResponseWrapper<String> response = this.client.put(uri, manifestData, Map.of("Content-Type", "application/vnd.oci.image.manifest.v1+json"), Scopes.of(this, containerRef, new Scope[0]), this.authProvider);
        this.logResponse(response);
        this.handleError(response);
        if (manifest.getSubject() != null && !response.headers().containsKey("OCI-Subject".toLowerCase())) {
            throw new OrasException("Subject was set on manifest but not OCI subject header was returned. Legacy flow not implemented");
        }
        return this.getManifest(containerRef);
    }

    @Override
    public Index pushIndex(ContainerRef containerRef, Index index) {
        URI uri = URI.create("%s://%s".formatted(this.getScheme(), containerRef.forRegistry(this).getManifestsPath(this)));
        byte[] indexData = JsonUtils.toJson(index).getBytes();
        LOG.debug("Index data to push: {}", (Object)new String(indexData, StandardCharsets.UTF_8));
        HttpClient.ResponseWrapper<String> response = this.client.put(uri, indexData, Map.of("Content-Type", "application/vnd.oci.image.index.v1+json"), Scopes.of(this, containerRef, new Scope[0]), this.authProvider);
        this.logResponse(response);
        this.handleError(response);
        return this.getIndex(containerRef);
    }

    public void deleteBlob(ContainerRef containerRef) {
        URI uri = URI.create("%s://%s".formatted(this.getScheme(), containerRef.forRegistry(this).getBlobsPath(this)));
        HttpClient.ResponseWrapper<String> response = this.client.delete(uri, Map.of(), Scopes.of(this, containerRef, new Scope[0]), this.authProvider);
        this.logResponse(response);
        this.handleError(response);
    }

    @Override
    public void pullArtifact(ContainerRef containerRef, Path path, boolean overwrite) {
        String contentType = this.getContentType(containerRef);
        List<Layer> layers = this.collectLayers(containerRef, contentType, false);
        if (layers.isEmpty()) {
            LOG.info("Skipped pulling layers without file name in '{}'", (Object)"org.opencontainers.image.title");
            return;
        }
        for (Layer layer : layers) {
            try {
                InputStream is = this.fetchBlob(containerRef.withDigest(layer.getDigest()));
                try {
                    if (Boolean.parseBoolean(layer.getAnnotations().getOrDefault("io.deis.oras.content.unpack", "false"))) {
                        LOG.debug("Extracting blob to: {}", (Object)path);
                        LocalPath tempArchive = ArchiveUtils.uncompress(is, layer.getMediaType());
                        String expectedDigest = layer.getAnnotations().get("io.deis.oras.content.digest");
                        if (expectedDigest != null) {
                            LOG.trace("Expected digest: {}", (Object)expectedDigest);
                            String actualDigest = containerRef.getAlgorithm().digest(tempArchive.getPath());
                            LOG.trace("Actual digest: {}", (Object)actualDigest);
                            if (!expectedDigest.equals(actualDigest)) {
                                throw new OrasException("Digest mismatch: expected %s but got %s".formatted(expectedDigest, actualDigest));
                            }
                        }
                        ArchiveUtils.untar(Files.newInputStream(tempArchive.getPath(), new OpenOption[0]), path);
                        continue;
                    }
                    Path targetPath = path.resolve(layer.getAnnotations().getOrDefault("org.opencontainers.image.title", layer.getDigest()));
                    if (Files.exists(targetPath, new LinkOption[0]) && !overwrite) {
                        LOG.info("File already exists: {}", (Object)targetPath);
                        continue;
                    }
                    LOG.debug("Copying blob to: {}", (Object)targetPath);
                    Files.copy(is, targetPath, StandardCopyOption.REPLACE_EXISTING);
                }
                finally {
                    if (is == null) continue;
                    is.close();
                }
            }
            catch (IOException e) {
                throw new OrasException("Failed to pull artifact", e);
            }
        }
    }

    @Override
    public Manifest pushArtifact(ContainerRef containerRef, ArtifactType artifactType, Annotations annotations, @Nullable Config config, LocalPath ... paths) {
        Manifest manifest = Manifest.empty().withArtifactType(artifactType);
        HashMap<String, String> manifestAnnotations = new HashMap<String, String>(annotations.manifestAnnotations());
        if (!manifestAnnotations.containsKey("org.opencontainers.image.created") && containerRef.getDigest() == null) {
            manifestAnnotations.put("org.opencontainers.image.created", Const.currentTimestamp());
        }
        manifest = manifest.withAnnotations(manifestAnnotations);
        if (config != null) {
            config = config.withAnnotations(annotations);
            manifest = manifest.withConfig(config);
        }
        List<Layer> layers = this.pushLayers(containerRef, false, paths);
        Config pushedConfig = this.pushConfig(containerRef, config != null ? config : Config.empty());
        manifest = manifest.withLayers(layers).withConfig(pushedConfig);
        manifest = this.pushManifest(containerRef, manifest);
        LOG.debug("Manifest pushed to: {}", (Object)containerRef.withDigest(manifest.getDescriptor().getDigest()));
        return manifest;
    }

    @Override
    public Layer pushBlob(ContainerRef containerRef, Path blob, Map<String, String> annotations) {
        String digest = containerRef.getAlgorithm().digest(blob);
        LOG.debug("Digest: {}", (Object)digest);
        if (this.hasBlob(containerRef.withDigest(digest))) {
            LOG.info("Blob already exists: {}", (Object)digest);
            return Layer.fromFile(blob, containerRef.getAlgorithm()).withAnnotations(annotations);
        }
        URI uri = URI.create("%s://%s".formatted(this.getScheme(), containerRef.withDigest(digest).forRegistry(this).getBlobsUploadDigestPath(this)));
        HttpClient.ResponseWrapper<String> response = this.client.upload("POST", uri, Map.of("Content-Type", "application/octet-stream"), blob, Scopes.of(this, containerRef, new Scope[0]), this.authProvider);
        this.logResponse(response);
        if (response.statusCode() == 201) {
            return Layer.fromFile(blob, containerRef.getAlgorithm()).withAnnotations(annotations);
        }
        if (response.statusCode() == 202) {
            String location = response.headers().get("Location".toLowerCase());
            if (!location.startsWith("http") && !location.startsWith("https")) {
                location = "%s://%s/%s".formatted(this.getScheme(), containerRef.getApiRegistry(this), location.replaceFirst("^/", ""));
            }
            LOG.debug("Location header: {}", (Object)location);
            URI uploadURI = this.createLocationWithDigest(location, digest);
            response = this.client.upload("PUT", uploadURI, Map.of("Content-Type", "application/octet-stream"), blob, Scopes.of(this, containerRef, new Scope[0]), this.authProvider);
            if (response.statusCode() == 201) {
                LOG.debug("Successful push: {}", (Object)response.response());
            } else {
                throw new OrasException("Failed to push layer: %s".formatted(response.response()));
            }
        }
        this.handleError(response);
        return Layer.fromFile(blob, containerRef.getAlgorithm()).withAnnotations(annotations);
    }

    @Override
    public Layer pushBlob(ContainerRef containerRef, byte[] data) {
        String digest = containerRef.getAlgorithm().digest(data);
        if (containerRef.getDigest() != null) {
            this.ensureDigest(containerRef, data);
        }
        if (this.hasBlob(containerRef.withDigest(digest))) {
            LOG.info("Blob already exists: {}", (Object)digest);
            return Layer.fromData(containerRef, data);
        }
        URI uri = URI.create("%s://%s".formatted(this.getScheme(), containerRef.withDigest(digest).forRegistry(this).getBlobsUploadDigestPath(this)));
        HttpClient.ResponseWrapper<String> response = this.client.post(uri, data, Map.of("Content-Type", "application/octet-stream"), Scopes.of(this, containerRef, new Scope[0]), this.authProvider);
        this.logResponse(response);
        if (response.statusCode() == 201) {
            return Layer.fromData(containerRef, data);
        }
        if (response.statusCode() == 202) {
            String location = response.headers().get("Location".toLowerCase());
            if (!location.startsWith("http") && !location.startsWith("https")) {
                location = "%s://%s/%s".formatted(this.getScheme(), containerRef.getApiRegistry(this), location.replaceFirst("^/", ""));
            }
            URI uploadURI = this.createLocationWithDigest(location, digest);
            LOG.debug("Location header: {}", (Object)location);
            response = this.client.put(uploadURI, data, Map.of("Content-Type", "application/octet-stream"), Scopes.of(this, containerRef, new Scope[0]), this.authProvider);
            if (response.statusCode() == 201) {
                LOG.debug("Successful push: {}", (Object)response.response());
            } else {
                throw new OrasException("Failed to push layer: %s".formatted(response.response()));
            }
        }
        this.handleError(response);
        return Layer.fromData(containerRef, data);
    }

    private boolean hasBlob(ContainerRef containerRef) {
        HttpClient.ResponseWrapper<String> response = this.headBlob(containerRef);
        return response.statusCode() == 200;
    }

    private HttpClient.ResponseWrapper<String> headBlob(ContainerRef containerRef) {
        URI uri = URI.create("%s://%s".formatted(this.getScheme(), containerRef.forRegistry(this).getBlobsPath(this)));
        HttpClient.ResponseWrapper<String> response = this.client.head(uri, Map.of("Accept", "application/octet-stream"), Scopes.of(this, containerRef, new Scope[0]), this.authProvider);
        this.logResponse(response);
        return response;
    }

    @Override
    public byte[] getBlob(ContainerRef containerRef) {
        if (!this.hasBlob(containerRef)) {
            throw new OrasException(new HttpClient.ResponseWrapper<String>("", 404, Map.of()));
        }
        URI uri = URI.create("%s://%s".formatted(this.getScheme(), containerRef.forRegistry(this).getBlobsPath(this)));
        HttpClient.ResponseWrapper<String> response = this.client.get(uri, Map.of("Accept", "application/octet-stream"), Scopes.of(this, containerRef, new Scope[0]), this.authProvider);
        this.logResponse(response);
        this.handleError(response);
        byte[] data = response.response().getBytes(StandardCharsets.UTF_8);
        this.validateDockerContentDigest(response, data);
        return data;
    }

    @Override
    public void fetchBlob(ContainerRef containerRef, Path path) {
        if (!this.hasBlob(containerRef)) {
            throw new OrasException(new HttpClient.ResponseWrapper<String>("", 404, Map.of()));
        }
        URI uri = URI.create("%s://%s".formatted(this.getScheme(), containerRef.forRegistry(this).getBlobsPath(this)));
        HttpClient.ResponseWrapper<Path> response = this.client.download(uri, Map.of("Accept", "application/octet-stream"), path, Scopes.of(this, containerRef, new Scope[0]), this.authProvider);
        this.logResponse(response);
        this.handleError(response);
        this.validateDockerContentDigest(response, path);
    }

    @Override
    public InputStream fetchBlob(ContainerRef containerRef) {
        if (!this.hasBlob(containerRef)) {
            throw new OrasException(new HttpClient.ResponseWrapper<String>("", 404, Map.of()));
        }
        URI uri = URI.create("%s://%s".formatted(this.getScheme(), containerRef.forRegistry(this).getBlobsPath(this)));
        HttpClient.ResponseWrapper<InputStream> response = this.client.download(uri, Map.of("Accept", "application/octet-stream"), Scopes.of(this, containerRef, new Scope[0]), this.authProvider);
        this.logResponse(response);
        this.handleError(response);
        this.validateDockerContentDigest(response);
        return response.response();
    }

    @Override
    public Descriptor fetchBlobDescriptor(ContainerRef containerRef) {
        HttpClient.ResponseWrapper<String> response = this.headBlob(containerRef);
        this.handleError(response);
        String size = response.headers().get("Content-Length".toLowerCase());
        return Descriptor.of(this.validateDockerContentDigest(response), Long.parseLong(size), "application/octet-stream");
    }

    @Override
    public Manifest getManifest(ContainerRef containerRef) {
        Descriptor descriptor = this.getDescriptor(containerRef);
        String contentType = descriptor.getMediaType();
        if (!this.isManifestMediaType(contentType)) {
            throw new OrasException("Expected manifest but got index. Probably a multi-platform image instead of artifact");
        }
        ManifestDescriptor manifestDescriptor = ManifestDescriptor.of(descriptor);
        return Manifest.fromJson(descriptor.getJson()).withDescriptor(manifestDescriptor);
    }

    @Override
    public Index getIndex(ContainerRef containerRef) {
        Descriptor descriptor = this.getDescriptor(containerRef);
        String contentType = descriptor.getMediaType();
        if (!this.isIndexMediaType(contentType)) {
            throw new OrasException("Expected index but got %s".formatted(contentType));
        }
        ManifestDescriptor manifestDescriptor = ManifestDescriptor.of(descriptor);
        return Index.fromJson(descriptor.getJson()).withDescriptor(manifestDescriptor);
    }

    @Override
    public Descriptor getDescriptor(ContainerRef containerRef) {
        HttpClient.ResponseWrapper<String> response = this.getManifestResponse(containerRef);
        this.handleError(response);
        String size = response.headers().get("Content-Length".toLowerCase());
        String contentType = response.headers().get("Content-Type".toLowerCase());
        return Descriptor.of(this.validateDockerContentDigest(response), Long.parseLong(size), contentType).withJson(response.response());
    }

    @Override
    public Descriptor probeDescriptor(ContainerRef ref) {
        Map<String, String> headers = this.getHeaders(ref);
        String digest = this.validateDockerContentDigest(headers);
        if (digest != null) {
            SupportedAlgorithm.fromDigest(digest);
        }
        String contentType = headers.get("Content-Type".toLowerCase());
        return Descriptor.of(digest, 0L, contentType);
    }

    boolean exists(ContainerRef containerRef) {
        URI uri = URI.create("%s://%s".formatted(this.getScheme(), containerRef.forRegistry(this).getManifestsPath(this)));
        HttpClient.ResponseWrapper<String> response = this.client.head(uri, Map.of("Accept", Const.MANIFEST_ACCEPT_TYPE), Scopes.of(this, containerRef, new Scope[0]), this.authProvider);
        this.logResponse(response);
        return response.statusCode() == 200;
    }

    private HttpClient.ResponseWrapper<String> getManifestResponse(ContainerRef containerRef) {
        URI uri = URI.create("%s://%s".formatted(this.getScheme(), containerRef.forRegistry(this).getManifestsPath(this)));
        HttpClient.ResponseWrapper<String> response = this.client.head(uri, Map.of("Accept", Const.MANIFEST_ACCEPT_TYPE), Scopes.of(this, containerRef, new Scope[0]), this.authProvider);
        this.logResponse(response);
        this.handleError(response);
        return this.client.get(uri, Map.of("Accept", Const.MANIFEST_ACCEPT_TYPE), Scopes.of(this, containerRef, new Scope[0]), this.authProvider);
    }

    private void validateDockerContentDigest(HttpClient.ResponseWrapper<String> response, byte[] data) {
        String digest = response.headers().get("Docker-Content-Digest".toLowerCase());
        if (digest == null) {
            LOG.warn("Docker-Content-Digest header not found in response. Skipping validation.");
            return;
        }
        String computedDigest = SupportedAlgorithm.fromDigest(digest).digest(data);
        this.ensureDigest(digest, computedDigest);
    }

    private void validateDockerContentDigest(HttpClient.ResponseWrapper<Path> response, Path path) {
        String digest = response.headers().get("Docker-Content-Digest".toLowerCase());
        if (digest == null) {
            LOG.warn("Docker-Content-Digest header not found in response. Skipping validation.");
            return;
        }
        String computedDigest = SupportedAlgorithm.fromDigest(digest).digest(path);
        this.ensureDigest(digest, computedDigest);
    }

    private @Nullable String validateDockerContentDigest(HttpClient.ResponseWrapper<?> response) {
        return this.validateDockerContentDigest(response.headers());
    }

    private @Nullable String validateDockerContentDigest(Map<String, String> headers) {
        String digest = headers.get("Docker-Content-Digest".toLowerCase());
        if (digest == null) {
            LOG.warn("Docker-Content-Digest header not found in response. Skipping validation.");
            return null;
        }
        SupportedAlgorithm.fromDigest(digest);
        return digest;
    }

    private void ensureDigest(ContainerRef ref, byte[] data) {
        if (ref.getDigest() == null) {
            throw new OrasException("Missing digest");
        }
        SupportedAlgorithm algorithm = SupportedAlgorithm.fromDigest(ref.getDigest());
        String dataDigest = algorithm.digest(data);
        this.ensureDigest(ref.getDigest(), dataDigest);
    }

    private void ensureDigest(String expected, @Nullable String current) {
        if (current == null) {
            throw new OrasException("Received null digest");
        }
        if (!expected.equals(current)) {
            throw new OrasException("Digest mismatch: %s != %s".formatted(expected, current));
        }
    }

    private void handleError(HttpClient.ResponseWrapper<?> responseWrapper) {
        if (responseWrapper.statusCode() >= 400) {
            if (responseWrapper.response() instanceof String) {
                LOG.debug("Response: {}", responseWrapper.response());
                throw new OrasException(responseWrapper);
            }
            throw new OrasException(new HttpClient.ResponseWrapper<String>("", responseWrapper.statusCode(), Map.of()));
        }
    }

    private void logResponse(HttpClient.ResponseWrapper<?> response) {
        LOG.debug("Status Code: {}", (Object)response.statusCode());
        LOG.debug("Headers: {}", response.headers());
        if (response.response() instanceof String) {
            LOG.debug("Response: {}", response.response());
        }
    }

    public InputStream getBlobStream(ContainerRef containerRef) {
        return this.fetchBlob(containerRef);
    }

    String getContentType(ContainerRef containerRef) {
        return this.getHeaders(containerRef).get("Content-Type".toLowerCase());
    }

    URI createLocationWithDigest(String location, String digest) {
        URI uploadURI;
        try {
            uploadURI = new URI(location);
            uploadURI = uploadURI.getQuery() == null ? new URI(String.valueOf(uploadURI) + "?digest=%s".formatted(digest)) : new URI(String.valueOf(uploadURI) + "&digest=%s".formatted(digest));
        }
        catch (URISyntaxException e) {
            throw new OrasException("Failed parse location header: %s".formatted(location));
        }
        return uploadURI;
    }

    Map<String, String> getHeaders(ContainerRef containerRef) {
        URI uri = URI.create("%s://%s".formatted(this.getScheme(), containerRef.forRegistry(this).getManifestsPath(this)));
        HttpClient.ResponseWrapper<String> response = this.client.head(uri, Map.of("Accept", Const.MANIFEST_ACCEPT_TYPE), Scopes.of(this, containerRef, new Scope[0]), this.authProvider);
        this.logResponse(response);
        this.handleError(response);
        return response.headers();
    }

    public static class Builder {
        private final Registry registry = new Registry();

        private Builder() {
        }

        public Builder defaults() {
            this.registry.setAuthProvider(new AuthStoreAuthenticationProvider());
            return this;
        }

        public Builder defaults(String registry) {
            return this.defaults().withRegistry(registry);
        }

        public Builder defaults(String username, String password) {
            this.registry.setAuthProvider(new UsernamePasswordProvider(username, password));
            return this;
        }

        public Builder defaults(String registry, String username, String password) {
            return this.defaults(username, password).withRegistry(registry);
        }

        public Builder insecure() {
            this.registry.setInsecure(true);
            this.registry.setSkipTlsVerify(true);
            this.registry.setAuthProvider(new NoAuthProvider());
            return this;
        }

        public Builder insecure(String registry) {
            return this.insecure().withRegistry(registry);
        }

        public Builder insecure(String registry, String username, String password) {
            return this.insecure().defaults(registry, username, password);
        }

        public Builder withRegistry(String registry) {
            this.registry.setRegistry(registry);
            return this;
        }

        public Builder withAuthProvider(AuthProvider authProvider) {
            this.registry.setAuthProvider(authProvider);
            return this;
        }

        public Builder withInsecure(boolean insecure) {
            this.registry.setInsecure(insecure);
            return this;
        }

        public Builder withSkipTlsVerify(boolean skipTlsVerify) {
            this.registry.setSkipTlsVerify(skipTlsVerify);
            return this;
        }

        public static Builder builder() {
            return new Builder();
        }

        public Registry build() {
            return this.registry.build();
        }
    }
}

