/*
 * Decompiled with CFR 0.152.
 */
package jenkins.plugins.openstack.compute.internal;

import com.cloudbees.plugins.credentials.common.PasswordCredentials;
import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.google.common.annotations.VisibleForTesting;
import hudson.Extension;
import hudson.ExtensionList;
import hudson.ExtensionPoint;
import hudson.Util;
import hudson.util.FormValidation;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.Optional;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import javax.annotation.CheckForNull;
import javax.annotation.Nonnegative;
import javax.annotation.Nonnull;
import javax.annotation.concurrent.ThreadSafe;
import jenkins.model.Jenkins;
import jenkins.plugins.openstack.compute.auth.OpenstackCredential;
import jenkins.plugins.openstack.compute.internal.FipScope;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.codec.digest.DigestUtils;
import org.jenkinsci.main.modules.instance_identity.InstanceIdentity;
import org.kohsuke.accmod.Restricted;
import org.kohsuke.accmod.restrictions.NoExternalUse;
import org.openstack4j.api.Builders;
import org.openstack4j.api.OSClient;
import org.openstack4j.api.client.IOSClientBuilder;
import org.openstack4j.api.compute.ServerService;
import org.openstack4j.api.exceptions.ResponseException;
import org.openstack4j.api.networking.NetFloatingIPService;
import org.openstack4j.api.networking.NetworkingService;
import org.openstack4j.core.transport.Config;
import org.openstack4j.model.common.ActionResponse;
import org.openstack4j.model.common.BasicResource;
import org.openstack4j.model.common.IdEntity;
import org.openstack4j.model.compute.Address;
import org.openstack4j.model.compute.Fault;
import org.openstack4j.model.compute.Flavor;
import org.openstack4j.model.compute.Keypair;
import org.openstack4j.model.compute.Server;
import org.openstack4j.model.compute.ServerCreate;
import org.openstack4j.model.compute.builder.ServerCreateBuilder;
import org.openstack4j.model.compute.ext.AvailabilityZone;
import org.openstack4j.model.identity.v2.Access;
import org.openstack4j.model.identity.v3.Token;
import org.openstack4j.model.image.v2.Image;
import org.openstack4j.model.network.ExternalGateway;
import org.openstack4j.model.network.NetFloatingIP;
import org.openstack4j.model.network.Network;
import org.openstack4j.model.network.Port;
import org.openstack4j.model.network.Router;
import org.openstack4j.model.network.ext.NetworkIPAvailability;
import org.openstack4j.model.network.options.PortListOptions;
import org.openstack4j.model.storage.block.Volume;
import org.openstack4j.model.storage.block.VolumeSnapshot;
import org.openstack4j.openstack.OSFactory;

@Restricted(value={NoExternalUse.class})
@ThreadSafe
public class Openstack {
    private static final Logger LOGGER = Logger.getLogger(Openstack.class.getName());
    public static final String FINGERPRINT_KEY_URL = "jenkins-instance";
    public static final String FINGERPRINT_KEY_FINGERPRINT = "jenkins-identity";
    private String INSTANCE_FINGERPRINT;
    private static final Comparator<Date> ACCEPT_NULLS = Comparator.nullsLast(Comparator.naturalOrder());
    private static final Comparator<Flavor> FLAVOR_COMPARATOR = Comparator.nullsLast(Comparator.comparing(Flavor::getName));
    private static final Comparator<AvailabilityZone> AVAILABILITY_ZONES_COMPARATOR = Comparator.nullsLast(Comparator.comparing(AvailabilityZone::getZoneName));
    private static final Comparator<VolumeSnapshot> VOLUMESNAPSHOT_DATE_COMPARATOR = Comparator.nullsLast(Comparator.comparing(VolumeSnapshot::getCreated, ACCEPT_NULLS).thenComparing(VolumeSnapshot::getId));
    private static final Comparator<Image> IMAGE_DATE_COMPARATOR = Comparator.nullsLast(Comparator.comparing(Image::getUpdatedAt, ACCEPT_NULLS).thenComparing(Image::getCreatedAt, ACCEPT_NULLS).thenComparing(Image::getId));
    private final ClientProvider clientProvider;
    @Nonnull
    private static final Cache<Openstack, List<? extends Network>> networksCache = Caffeine.newBuilder().expireAfterWrite(10L, TimeUnit.MINUTES).build();
    @Nonnull
    private static final Cache<Openstack, List<? extends NetworkIPAvailability>> networkIpAvailabilityCache = Caffeine.newBuilder().expireAfterWrite(5L, TimeUnit.SECONDS).build();

    private Openstack(@Nonnull String endPointUrl, boolean ignoreSsl, @Nonnull OpenstackCredential auth, @CheckForNull String region, @Nonnull long cleanfreq) {
        IOSClientBuilder<? extends OSClient<?>, ?> builder = auth.getBuilder(endPointUrl);
        Config config = Config.newConfig();
        config.withConnectionTimeout(20000);
        config.withReadTimeout(20000);
        if (ignoreSsl) {
            config.withSSLVerificationDisabled();
        }
        OSClient client = ((OSClient)builder.withConfig(config).authenticate()).useRegion(region);
        this.clientProvider = ClientProvider.get(client, region, config);
        Openstack.debug("Openstack client created for \"{0}\", \"{1}\".", auth.toString(), region);
    }

    @VisibleForTesting
    public Openstack(final @Nonnull OSClient<?> client) {
        this.clientProvider = new ClientProvider(){

            @Override
            @Nonnull
            public OSClient<?> get() {
                return client;
            }

            @Override
            @Nonnull
            public String getInfo() {
                return "";
            }
        };
    }

    @Nonnull
    public static String getFlavorInfo(@Nonnull Flavor f) {
        return String.format("%s (CPUs: %s, RAM: %sMB, Disk: %sGB, SWAP: %sMB, Ephemeral: %sGB)", f.getName(), f.getVcpus(), f.getRam(), f.getDisk(), f.getSwap(), f.getEphemeral());
    }

    @Nonnull
    public String getInfo() {
        return this.clientProvider.getInfo();
    }

    @Nonnull
    @VisibleForTesting
    public List<? extends Network> _listNetworks() {
        return Objects.requireNonNull((List)networksCache.get((Object)this, os -> os.clientProvider.get().networking().network().list()));
    }

    @Nonnull
    public Map<String, Network> getNetworks(@Nonnull List<String> nameOrIds) {
        if (nameOrIds.isEmpty()) {
            return Collections.emptyMap();
        }
        nameOrIds = new ArrayList<String>(nameOrIds);
        HashMap<String, Network> ret = new HashMap<String, Network>();
        List<? extends Network> networks = this._listNetworks();
        for (Network network : networks) {
            if (nameOrIds.contains(network.getName())) {
                ret.put(network.getId(), network);
                nameOrIds.removeAll(Collections.singletonList(network.getName()));
            } else if (nameOrIds.contains(network.getId())) {
                ret.put(network.getId(), network);
                nameOrIds.removeAll(Collections.singletonList(network.getId()));
            }
            if (!nameOrIds.isEmpty()) continue;
            break;
        }
        if (!nameOrIds.isEmpty()) {
            throw new NoSuchElementException("Unable to find networks for: " + String.valueOf(nameOrIds));
        }
        return ret;
    }

    @Nonnull
    public Map<Network, Integer> getNetworksCapacity(@Nonnull Map<String, Network> declaredNetworks) {
        if (declaredNetworks.isEmpty()) {
            throw new IllegalArgumentException("No request networks provided");
        }
        List declaredIds = declaredNetworks.values().stream().map(IdEntity::getId).collect(Collectors.toList());
        List<? extends NetworkIPAvailability> networkIPAvailabilities = this.getNetworkIPAvailability();
        return networkIPAvailabilities.stream().filter(n -> declaredIds.contains(n.getNetworkId())).collect(Collectors.toMap(n -> (Network)declaredNetworks.get(n.getNetworkId()), n -> n.getTotalIps().subtract(n.getUsedIps()).intValue()));
    }

    public List<? extends NetworkIPAvailability> getNetworkIPAvailability() {
        return Objects.requireNonNull((List)networkIpAvailabilityCache.get((Object)this, os -> os.clientProvider.get().networking().networkIPAvailability().get()));
    }

    @Nonnull
    public Map<String, List<Image>> getImages() {
        List<Image> list = this.getAllImages();
        TreeMap<String, List<Image>> data = new TreeMap<String, List<Image>>(String.CASE_INSENSITIVE_ORDER);
        for (Image image : list) {
            String name = Util.fixNull((String)image.getName());
            String nameOrId = name.isEmpty() ? image.getId() : name;
            List<Image> sameNamed = data.get(nameOrId);
            if (sameNamed == null) {
                sameNamed = new ArrayList<Image>();
                data.putIfAbsent(nameOrId, sameNamed);
            }
            sameNamed.add(image);
        }
        for (List sameNamed : data.values()) {
            sameNamed.sort(IMAGE_DATE_COMPARATOR);
        }
        return data;
    }

    @Nonnull
    private List<Image> getAllImages() {
        int LIMIT = 100;
        HashMap<String, String> params = new HashMap<String, String>(2);
        params.put("limit", Integer.toString(100));
        List page = this.clientProvider.get().imagesV2().list(params);
        ArrayList<Image> all = new ArrayList<Image>(page);
        while (page.size() == 100) {
            params.put("marker", ((Image)page.get(99)).getId());
            page = this.clientProvider.get().imagesV2().list(params);
            all.addAll(page);
        }
        return all;
    }

    @Nonnull
    public Map<String, List<VolumeSnapshot>> getVolumeSnapshots() {
        List list = this.clientProvider.get().blockStorage().snapshots().list();
        TreeMap<String, List<VolumeSnapshot>> data = new TreeMap<String, List<VolumeSnapshot>>(String.CASE_INSENSITIVE_ORDER);
        for (VolumeSnapshot vs : list) {
            if (vs.getStatus() != Volume.Status.AVAILABLE) continue;
            String name = Util.fixNull((String)vs.getName());
            String nameOrId = name.isEmpty() ? vs.getId() : name;
            List<VolumeSnapshot> sameNamed = data.get(nameOrId);
            if (sameNamed == null) {
                sameNamed = new ArrayList<VolumeSnapshot>();
                data.putIfAbsent(nameOrId, sameNamed);
            }
            sameNamed.add(vs);
        }
        for (List sameNamed : data.values()) {
            sameNamed.sort(VOLUMESNAPSHOT_DATE_COMPARATOR);
        }
        return data;
    }

    @Nonnull
    public Collection<? extends Flavor> getSortedFlavors() {
        List flavors = this.clientProvider.get().compute().flavors().list();
        flavors.sort(FLAVOR_COMPARATOR);
        return flavors;
    }

    @Nonnull
    public List<String> getSortedIpPools() {
        List routers = this.clientProvider.get().networking().router().list();
        List networks = this.clientProvider.get().networking().network().list();
        HashSet<Network> applicablePublicNetworks = new HashSet<Network>();
        for (Router r : routers) {
            for (Network n : networks) {
                ExternalGateway gateway = r.getExternalGatewayInfo();
                if (gateway == null || !Objects.equals(gateway.getNetworkId(), n.getId())) continue;
                applicablePublicNetworks.add(n);
            }
        }
        return applicablePublicNetworks.stream().map(BasicResource::getName).sorted().collect(Collectors.toList());
    }

    @Nonnull
    public List<? extends AvailabilityZone> getAvailabilityZones() {
        List zones = this.clientProvider.get().compute().zones().list();
        zones.sort(AVAILABILITY_ZONES_COMPARATOR);
        return zones;
    }

    @Nonnull
    public List<Server> getRunningNodes() {
        ArrayList<Server> running = new ArrayList<Server>();
        boolean detailed = true;
        for (Server n : this.clientProvider.get().compute().servers().list(true)) {
            if (!Openstack.isOccupied(n) || !this.isOurs(n)) continue;
            running.add(n);
        }
        return running;
    }

    @Nonnull
    public List<String> getFreeFipIds() {
        ArrayList<String> freeIps = new ArrayList<String>();
        for (NetFloatingIP ip : this.clientProvider.get().networking().floatingip().list()) {
            String serverId;
            if (ip.getFixedIpAddress() != null || (serverId = FipScope.getServerId(this.instanceUrl(), this.instanceFingerprint(), ip.getDescription())) == null) continue;
            freeIps.add(ip.getId());
        }
        return freeIps;
    }

    @Nonnull
    public List<String> getSortedKeyPairNames() {
        ArrayList<String> keyPairs = new ArrayList<String>();
        for (Keypair kp : this.clientProvider.get().compute().keypairs().list()) {
            keyPairs.add(kp.getName());
        }
        return keyPairs;
    }

    @Nonnull
    public List<String> getImageIdsFor(String nameOrId) {
        Image findById;
        TreeSet<Image> sortedObjects = new TreeSet<Image>(IMAGE_DATE_COMPARATOR);
        HashMap<String, String> query = new HashMap<String, String>(2);
        query.put("name", nameOrId);
        query.put("status", "active");
        List findByName = this.clientProvider.get().imagesV2().list(query);
        sortedObjects.addAll(findByName);
        if (nameOrId.matches("[0-9a-f-]{36}") && (findById = this.clientProvider.get().imagesV2().get(nameOrId)) != null && findById.getStatus() == Image.ImageStatus.ACTIVE) {
            sortedObjects.add(findById);
        }
        ArrayList<String> ids = new ArrayList<String>();
        for (Image i : sortedObjects) {
            ids.add(i.getId());
        }
        return ids;
    }

    @Nonnull
    public List<String> getVolumeSnapshotIdsFor(String nameOrId) {
        VolumeSnapshot findById;
        TreeSet<VolumeSnapshot> sortedObjects = new TreeSet<VolumeSnapshot>(VOLUMESNAPSHOT_DATE_COMPARATOR);
        Map<String, List<VolumeSnapshot>> allVolumeSnapshots = this.getVolumeSnapshots();
        Collection findByName = allVolumeSnapshots.get(nameOrId);
        if (findByName != null) {
            sortedObjects.addAll(findByName);
        }
        if (nameOrId.matches("[0-9a-f-]{36}") && (findById = this.clientProvider.get().blockStorage().snapshots().get(nameOrId)) != null && findById.getStatus() == Volume.Status.AVAILABLE) {
            sortedObjects.add(findById);
        }
        ArrayList<String> ids = new ArrayList<String>();
        for (VolumeSnapshot i : sortedObjects) {
            ids.add(i.getId());
        }
        return ids;
    }

    @CheckForNull
    public String getVolumeSnapshotDescription(String volumeSnapshotId) {
        return this.clientProvider.get().blockStorage().snapshots().get(volumeSnapshotId).getDescription();
    }

    public void setVolumeNameAndDescription(String volumeId, String newVolumeName, String newVolumeDescription) {
        ActionResponse res = this.clientProvider.get().blockStorage().volumes().update(volumeId, newVolumeName, newVolumeDescription);
        Openstack.throwIfFailed(res);
    }

    public static boolean isOccupied(@Nonnull Server server) {
        switch (server.getStatus()) {
            case UNKNOWN: 
            case MIGRATING: 
            case SHUTOFF: 
            case DELETED: {
                return false;
            }
            case UNRECOGNIZED: {
                LOGGER.log(Level.WARNING, "Machine state not recognized by openstack4j, report this as a bug: " + String.valueOf(server));
                return true;
            }
        }
        return true;
    }

    private boolean isOurs(@Nonnull Server server) {
        Map metadata = server.getMetadata();
        String serverFingerprint = (String)metadata.get(FINGERPRINT_KEY_FINGERPRINT);
        return serverFingerprint == null ? Objects.equals(this.instanceUrl(), metadata.get(FINGERPRINT_KEY_URL)) : Objects.equals(this.instanceUrl(), metadata.get(FINGERPRINT_KEY_URL)) && Objects.equals(this.instanceFingerprint(), serverFingerprint);
    }

    @Nonnull
    @VisibleForTesting
    public String instanceUrl() {
        String rootUrl = Jenkins.get().getRootUrl();
        if (rootUrl == null) {
            throw new IllegalStateException("Jenkins instance URL is not configured");
        }
        return rootUrl;
    }

    @Nonnull
    @VisibleForTesting
    public String instanceFingerprint() {
        if (this.INSTANCE_FINGERPRINT == null) {
            this.INSTANCE_FINGERPRINT = DigestUtils.sha1Hex((String)("openstack-cloud-plugin-identity-fingerprint:" + new String(Base64.encodeBase64((byte[])InstanceIdentity.get().getPublic().getEncoded()), StandardCharsets.UTF_8)));
        }
        return this.INSTANCE_FINGERPRINT;
    }

    @Nonnull
    public Server getServerById(@Nonnull String id) throws NoSuchElementException {
        Server server = this.clientProvider.get().compute().servers().get(id);
        if (server == null) {
            throw new NoSuchElementException("No such server running: " + id);
        }
        return server;
    }

    @Nonnull
    public List<Server> getServersByName(@Nonnull String name) {
        ArrayList<Server> ret = new ArrayList<Server>();
        for (Server server : this.clientProvider.get().compute().servers().list(Collections.singletonMap("name", name))) {
            if (!this.isOurs(server)) continue;
            ret.add(server);
        }
        return ret;
    }

    @Nonnull
    public Server bootAndWaitActive(@Nonnull ServerCreateBuilder request, @Nonnegative int timeout) throws ActionFailed {
        Openstack.debug("Booting machine", new String[0]);
        this.attachFingerprint(request);
        try {
            Server server = this._bootAndWaitActive(request, timeout);
            if (server == null) {
                String name = ((ServerCreate)request.build()).getName();
                List<Server> servers = this.getServersByName(name);
                String msg = "Failed to provision the " + name + " in time (" + timeout + "ms). Existing server(s): " + servers.toString();
                ActionFailed err = new ActionFailed(msg);
                try {
                    int size = servers.size();
                    if (size == 1) {
                        this.destroyServer(servers.get(0));
                    } else if (size > 1) {
                        LOGGER.warning("Unable to destroy server " + name + " as there is " + size + " of them");
                    }
                }
                catch (Throwable ex) {
                    err.addSuppressed(ex);
                }
                throw err;
            }
            Openstack.debug("Machine started: {0}", server.getName());
            this.throwIfFailed(server);
            return server;
        }
        catch (ResponseException ex) {
            throw new ActionFailed(ex.getMessage(), ex);
        }
    }

    @VisibleForTesting
    public void attachFingerprint(@Nonnull ServerCreateBuilder request) {
        request.addMetadataItem(FINGERPRINT_KEY_URL, this.instanceUrl());
        request.addMetadataItem(FINGERPRINT_KEY_FINGERPRINT, this.instanceFingerprint());
    }

    @Restricted(value={NoExternalUse.class})
    public Server _bootAndWaitActive(@Nonnull ServerCreateBuilder request, @Nonnegative int timeout) {
        return this.clientProvider.get().compute().servers().bootAndWaitActive((ServerCreate)request.build(), timeout);
    }

    @Nonnull
    public Server updateInfo(@Nonnull Server server) {
        return this.getServerById(server.getId());
    }

    public void destroyServer(@Nonnull Server server) throws ActionFailed {
        ActionResponse serverDelete;
        String nodeId = server.getId();
        NetFloatingIPService fipService = this.clientProvider.get().networking().floatingip();
        List portIds = this.getServerPorts(server).stream().map(IdEntity::getId).collect(Collectors.toList());
        List associatedFips = fipService.list().stream().filter(fip -> portIds.contains(fip.getPortId())).collect(Collectors.toList());
        ServerService servers = this.clientProvider.get().compute().servers();
        server = servers.get(nodeId);
        if (server == null || server.getStatus() == Server.Status.DELETED) {
            Openstack.debug("Machine destroyed: {0}", nodeId);
        }
        if ((serverDelete = servers.delete(nodeId)).getCode() == 404) {
            Openstack.debug("Machine destroyed: {0}", nodeId);
        } else {
            Openstack.throwIfFailed(serverDelete);
        }
        for (NetFloatingIP fip2 : associatedFips) {
            ActionResponse fipDelete = fipService.delete(fip2.getId());
            if (fipDelete.getCode() == 404) {
                Openstack.debug("Fip destroyed: {0}", fip2.getId());
                continue;
            }
            Openstack.throwIfFailed(fipDelete);
        }
    }

    @Nonnull
    public Server assignFloatingIp(@Nonnull Server server, @Nonnull String poolName) throws ActionFailed {
        Openstack.debug("Allocating floating IP for {0} in {1}", server.getName(), server.getName());
        NetworkingService networking = this.clientProvider.get().networking();
        String desc = FipScope.getDescription(this.instanceUrl(), this.instanceFingerprint(), server);
        Port port = this.getServerPorts(server).get(0);
        Network network = (Network)networking.network().list(Collections.singletonMap("name", poolName)).get(0);
        NetFloatingIP fip = (NetFloatingIP)Builders.netFloatingIP().floatingNetworkId(network.getId()).portId(port.getId()).description(desc).build();
        try {
            NetFloatingIP ip = networking.floatingip().create(fip);
            for (int i = 0; i < 30; ++i) {
                try {
                    Thread.sleep(1000L);
                }
                catch (InterruptedException ex) {
                    Thread.currentThread().interrupt();
                    throw new ActionFailed("Interrupted", ex);
                }
                server = this.updateInfo(server);
                Optional<Address> newIp = server.getAddresses().getAddresses().values().stream().flatMap(Collection::stream).filter(address -> Objects.equals(address.getAddr(), ip.getFloatingIpAddress())).findFirst();
                if (!newIp.isPresent()) continue;
                return server;
            }
            this.destroyFip(ip.getId());
            throw new ActionFailed("IP address not propagated in time for " + server.getName());
        }
        catch (ResponseException ex) {
            throw new ActionFailed(ex.getMessage() + " Allocating for " + server.getName(), ex);
        }
    }

    private List<? extends Port> getServerPorts(@Nonnull Server server) {
        return this.clientProvider.get().networking().port().list(PortListOptions.create().deviceId(server.getId()));
    }

    public void destroyFip(String fip) {
        ActionResponse delete = this.clientProvider.get().networking().floatingip().delete(fip);
        if (delete.getCode() == 404) {
            return;
        }
        Openstack.throwIfFailed(delete);
    }

    @CheckForNull
    public static String getAccessIpAddress(@Nonnull Server server) throws IllegalArgumentException, NoSuchElementException {
        return Openstack.getAccessIpAddressObject(server).getAddr();
    }

    public static Address getAccessIpAddressObject(@Nonnull Server server) {
        Address fixedIPv4 = null;
        Address fixedIPv6 = null;
        Address floatingIPv6 = null;
        Collection addressMap = server.getAddresses().getAddresses().values();
        for (List addresses : addressMap) {
            for (Address addr : addresses) {
                String type = addr.getType();
                int version = addr.getVersion();
                String address = addr.getAddr();
                if (version != 4 && version != 6) {
                    throw new IllegalArgumentException("Unknown or unsupported IP protocol version: " + version);
                }
                if (Objects.equals(type, "floating")) {
                    if (version == 4) {
                        return addr;
                    }
                    if (floatingIPv6 != null) continue;
                    floatingIPv6 = addr;
                    continue;
                }
                if (version == 4) {
                    if (fixedIPv4 != null) continue;
                    fixedIPv4 = addr;
                    continue;
                }
                if (fixedIPv6 != null) continue;
                fixedIPv6 = addr;
            }
        }
        if (floatingIPv6 != null) {
            return floatingIPv6;
        }
        if (fixedIPv4 != null) {
            return fixedIPv4;
        }
        if (fixedIPv6 != null) {
            return fixedIPv6;
        }
        throw new NoSuchElementException("No access IP address found for " + server.getName() + ": " + String.valueOf(addressMap));
    }

    private static void throwIfFailed(@Nonnull ActionResponse res) {
        if (res.isSuccess()) {
            return;
        }
        throw new ActionFailed(res.toString());
    }

    private void throwIfFailed(@Nonnull Server server) {
        Server.Status status = server.getStatus();
        if (status == Server.Status.ACTIVE) {
            return;
        }
        StringBuilder sb = new StringBuilder();
        sb.append("Failed to boot server ").append(server.getName());
        if (status == Server.Status.BUILD) {
            sb.append(" in time:");
        } else {
            sb.append(":");
        }
        sb.append(" status=").append(status);
        sb.append(" vmState=").append(server.getVmState());
        Fault fault = server.getFault();
        String msg = fault == null ? "none" : String.format("%d: %s (%s)", fault.getCode(), fault.getMessage(), fault.getDetails());
        sb.append(" fault=").append(msg);
        ActionFailed ex = new ActionFailed(sb.toString());
        try {
            this.destroyServer(server);
        }
        catch (ActionFailed suppressed) {
            ex.addSuppressed(suppressed);
        }
        LOGGER.log(Level.WARNING, "Machine provisioning failed: " + String.valueOf(server), ex);
        throw ex;
    }

    @CheckForNull
    public Throwable sanityCheck() {
        try {
            OSClient<?> client = this.clientProvider.get();
            client.networking().network().list();
            client.images().listMembers("");
            client.compute().flavors().list();
        }
        catch (Throwable ex) {
            return ex;
        }
        return null;
    }

    private static void debug(@Nonnull String msg, String ... args) {
        LOGGER.log(Level.FINE, msg, args);
    }

    @Nonnull
    private static String getCloudConnectionFingerprint(@Nonnull String endPointUrl, boolean ignoreSsl, @Nonnull OpenstackCredential auth, @CheckForNull String region) {
        String pwd = auth instanceof PasswordCredentials ? ((PasswordCredentials)auth).getPassword().getEncryptedValue() : "";
        String plain = String.join((CharSequence)"::", endPointUrl, Boolean.toString(ignoreSsl), auth.toString(), pwd, region);
        return Util.getDigestOf((String)plain);
    }

    private static abstract class ClientProvider {
        private ClientProvider() {
        }

        @Nonnull
        public abstract OSClient<?> get();

        @Nonnull
        public abstract String getInfo();

        private static ClientProvider get(OSClient<?> client, String region, Config config) {
            if (client instanceof OSClient.OSClientV2) {
                return new SessionClientV2Provider((OSClient.OSClientV2)client, region, config);
            }
            if (client instanceof OSClient.OSClientV3) {
                return new SessionClientV3Provider((OSClient.OSClientV3)client, region, config);
            }
            throw new AssertionError((Object)("Unsupported openstack4j client " + client.getClass().getName()));
        }

        private static class SessionClientV2Provider
        extends ClientProvider {
            protected final Access storage;
            protected final String region;
            protected final Config config;

            private SessionClientV2Provider(OSClient.OSClientV2 toStore, String usedRegion, Config clientConfig) {
                this.storage = toStore.getAccess();
                this.region = usedRegion;
                this.config = clientConfig;
            }

            @Override
            @Nonnull
            public OSClient<?> get() {
                return OSFactory.clientFromAccess((Access)this.storage, (Config)this.config).useRegion(this.region);
            }

            @Override
            @Nonnull
            public String getInfo() {
                StringBuilder sb = new StringBuilder();
                for (Access.Service service : this.storage.getServiceCatalog()) {
                    if (sb.length() != 0) {
                        sb.append(", ");
                    }
                    sb.append(service.getType()).append('/').append(service.getName()).append(':').append(service.getVersion());
                }
                return sb.toString();
            }
        }

        private static class SessionClientV3Provider
        extends ClientProvider {
            private final Token storage;
            private final String region;
            protected final Config config;

            private SessionClientV3Provider(OSClient.OSClientV3 toStore, String usedRegion, Config clientConfig) {
                this.storage = toStore.getToken();
                this.region = usedRegion;
                this.config = clientConfig;
            }

            @Override
            @Nonnull
            public OSClient<?> get() {
                return OSFactory.clientFromToken((Token)this.storage, (Config)this.config).useRegion(this.region);
            }

            @Override
            @Nonnull
            public String getInfo() {
                return "";
            }
        }
    }

    public static final class ActionFailed
    extends RuntimeException {
        private static final long serialVersionUID = -1657469882396520333L;

        public ActionFailed(String msg) {
            super(msg);
        }

        public ActionFailed(String msg, Throwable cause) {
            super(msg, cause);
        }
    }

    @Extension
    public static final class Factory
    extends FactoryEP {
        @Override
        @Nonnull
        public Openstack getOpenstack(@Nonnull String endPointUrl, boolean ignoreSsl, @Nonnull OpenstackCredential auth, @CheckForNull String region, @Nonnull long cleanfreq) throws FormValidation {
            endPointUrl = Util.fixEmptyAndTrim((String)endPointUrl);
            region = Util.fixEmptyAndTrim((String)region);
            if (endPointUrl == null) {
                throw FormValidation.error((String)"No endPoint specified");
            }
            if (auth == null) {
                throw FormValidation.error((String)"No credential specified");
            }
            return new Openstack(endPointUrl, ignoreSsl, auth, region, cleanfreq);
        }
    }

    @Restricted(value={NoExternalUse.class})
    public static abstract class FactoryEP
    implements ExtensionPoint {
        @Nonnull
        private final transient Cache<String, Openstack> cache = Caffeine.newBuilder().expireAfterWrite(10L, TimeUnit.MINUTES).build();

        @Nonnull
        public abstract Openstack getOpenstack(@Nonnull String var1, boolean var2, @Nonnull OpenstackCredential var3, @CheckForNull String var4, @Nonnull long var5) throws FormValidation;

        @Nonnull
        public static Openstack get(@Nonnull String endPointUrl, boolean ignoreSsl, @Nonnull OpenstackCredential auth, @CheckForNull String region, @Nonnull long cleanfreq) throws FormValidation {
            FactoryEP ep = (FactoryEP)ExtensionList.lookup(FactoryEP.class).get(0);
            Function<String, Openstack> cacheMissFunction = unused -> {
                try {
                    return ep.getOpenstack(endPointUrl, ignoreSsl, auth, region, cleanfreq);
                }
                catch (FormValidation ex) {
                    throw new RuntimeException(ex);
                }
            };
            String fingerprint = Openstack.getCloudConnectionFingerprint(endPointUrl, ignoreSsl, auth, region);
            try {
                return Objects.requireNonNull((Openstack)ep.cache.get((Object)fingerprint, cacheMissFunction));
            }
            catch (RuntimeException e) {
                Throwable cause = e.getCause();
                if (cause instanceof FormValidation) {
                    throw (FormValidation)cause;
                }
                if (cause instanceof RuntimeException) {
                    throw (RuntimeException)cause;
                }
                throw e;
            }
        }

        @Nonnull
        public static FactoryEP replace(@Nonnull FactoryEP factory) {
            ExtensionList lookup = ExtensionList.lookup(FactoryEP.class);
            lookup.clear();
            lookup.add((Object)factory);
            return factory;
        }

        @Restricted(value={NoExternalUse.class})
        @Nonnull
        public static Cache<String, Openstack> getCache() {
            FactoryEP ep = (FactoryEP)ExtensionList.lookup(FactoryEP.class).get(0);
            return ep.cache;
        }
    }
}

