/*
 * Decompiled with CFR 0.152.
 */
package org.csanchez.jenkins.plugins.kubernetes;

import edu.umd.cs.findbugs.annotations.CheckForNull;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import hudson.Functions;
import hudson.model.Descriptor;
import hudson.model.Run;
import hudson.model.TaskListener;
import hudson.slaves.ComputerLauncher;
import hudson.slaves.JNLPLauncher;
import hudson.slaves.SlaveComputer;
import io.fabric8.kubernetes.api.model.ContainerStatus;
import io.fabric8.kubernetes.api.model.ObjectMeta;
import io.fabric8.kubernetes.api.model.Pod;
import io.fabric8.kubernetes.client.KubernetesClient;
import io.fabric8.kubernetes.client.KubernetesClientException;
import io.fabric8.kubernetes.client.dsl.ContainerResource;
import io.fabric8.kubernetes.client.dsl.NonNamespaceOperation;
import io.fabric8.kubernetes.client.dsl.PodResource;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import jenkins.metrics.api.Metrics;
import jenkins.util.SystemProperties;
import org.apache.commons.lang.StringUtils;
import org.csanchez.jenkins.plugins.kubernetes.KubernetesCloud;
import org.csanchez.jenkins.plugins.kubernetes.KubernetesComputer;
import org.csanchez.jenkins.plugins.kubernetes.KubernetesSlave;
import org.csanchez.jenkins.plugins.kubernetes.MetricNames;
import org.csanchez.jenkins.plugins.kubernetes.PodTemplate;
import org.csanchez.jenkins.plugins.kubernetes.PodUtils;
import org.csanchez.jenkins.plugins.kubernetes.pod.decorator.PodDecoratorException;
import org.csanchez.jenkins.plugins.kubernetes.pod.retention.Reaper;
import org.kohsuke.stapler.DataBoundConstructor;

public class KubernetesLauncher
extends JNLPLauncher {
    private static final long REPORT_INTERVAL = TimeUnit.SECONDS.toMillis(30L);
    private static final Collection<String> POD_TERMINATED_STATES = Collections.unmodifiableCollection(Arrays.asList("Succeeded", "Failed"));
    private static final Logger LOGGER = Logger.getLogger(KubernetesLauncher.class.getName());
    private volatile boolean launched = false;
    private static final boolean DISABLE_DIAGNOSTIC_LOGS = SystemProperties.getBoolean((String)(KubernetesLauncher.class.getName() + ".disableDiagnosticLogs"), (boolean)false);
    @CheckForNull
    private transient Throwable problem;

    @DataBoundConstructor
    public KubernetesLauncher(String tunnel, String vmargs) {
        super(tunnel, vmargs);
    }

    public KubernetesLauncher() {
    }

    public boolean isLaunchSupported() {
        return !this.launched;
    }

    @SuppressFBWarnings(value={"SWL_SLEEP_WITH_LOCK_HELD"}, justification="This is fine")
    public synchronized void launch(SlaveComputer computer, TaskListener listener) {
        if (!(computer instanceof KubernetesComputer)) {
            throw new IllegalArgumentException("This Launcher can be used only with KubernetesComputer");
        }
        Reaper.getInstance().maybeActivate();
        KubernetesComputer kubernetesComputer = (KubernetesComputer)computer;
        computer.setAcceptingTasks(false);
        KubernetesSlave node = (KubernetesSlave)kubernetesComputer.getNode();
        if (node == null) {
            throw new IllegalStateException("Node has been removed, cannot launch " + computer.getName());
        }
        if (this.launched) {
            LOGGER.log(Level.INFO, "Agent has already been launched, activating: {0}", node.getNodeName());
            computer.setAcceptingTasks(true);
            return;
        }
        String cloudName = node.getCloudName();
        try {
            int waitedForSlave;
            Pod existingPod;
            Pod pod;
            PodTemplate template = node.getTemplate();
            KubernetesCloud cloud = node.getKubernetesCloud();
            KubernetesClient client = cloud.connect();
            try {
                pod = template.build(node);
            }
            catch (PodDecoratorException e) {
                Run<?, ?> run = template.getRun();
                if (run != null) {
                    template.getListener().getLogger().println("Failed to build pod definition : " + e.getMessage());
                    PodUtils.cancelQueueItemFor(run.getUrl(), template.getLabel(), e.getMessage(), null);
                }
                e.printStackTrace(listener.fatalError("Failed to build pod definition"));
                this.setProblem(e);
                KubernetesLauncher.terminateOrLog(node);
                return;
            }
            node.assignPod(pod);
            String podName = pod.getMetadata().getName();
            String namespace = Arrays.asList(pod.getMetadata().getNamespace(), template.getNamespace(), client.getNamespace()).stream().filter(s -> StringUtils.isNotBlank((String)s)).findFirst().orElse(null);
            node.setNamespace(namespace);
            if (!DISABLE_DIAGNOSTIC_LOGS) {
                cloud.registerPodInformer(node);
            }
            if ((existingPod = (Pod)((PodResource)((NonNamespaceOperation)client.pods().inNamespace(namespace)).withName(podName)).get()) == null) {
                LOGGER.log(Level.FINE, () -> "Creating Pod: " + cloudName + " " + namespace + "/" + podName);
                try {
                    pod = (Pod)((NonNamespaceOperation)client.pods().inNamespace(namespace)).create((Object)pod);
                }
                catch (KubernetesClientException e) {
                    Metrics.metricRegistry().counter("kubernetes.cloud.pods.creation.failed").inc();
                    int httpCode = e.getCode();
                    if (400 <= httpCode && httpCode < 500) {
                        if (httpCode == 403 && e.getMessage().contains("is forbidden: exceeded quota")) {
                            node.getRunListener().getLogger().printf("WARNING: Unable to create pod: %s %s/%s because kubernetes resource quota exceeded. %n%s%nRetrying...%n%n", cloudName, namespace, pod.getMetadata().getName(), e.getMessage());
                        } else if (httpCode == 409 && e.getMessage().contains("Operation cannot be fulfilled on resourcequotas")) {
                            node.getRunListener().getLogger().printf("WARNING: Unable to create pod: %s %s/%s because kubernetes resource quota update conflict. %n%s%nRetrying...%n%n", cloudName, namespace, pod.getMetadata().getName(), e.getMessage());
                        } else {
                            node.getRunListener().getLogger().printf("ERROR: Unable to create pod %s %s/%s.%n%s%n", cloudName, namespace, pod.getMetadata().getName(), e.getMessage());
                            PodUtils.cancelQueueItemFor(pod, e.getMessage());
                        }
                    } else if (500 <= httpCode && httpCode < 600) {
                        LOGGER.log(Level.FINE, "Kubernetes returned HTTP code {0} {1}. Retrying...", new Object[]{e.getCode(), e.getStatus()});
                    } else {
                        LOGGER.log(Level.WARNING, "Kubernetes returned unhandled HTTP code {0} {1}", new Object[]{e.getCode(), e.getStatus()});
                    }
                    throw e;
                }
                LOGGER.log(Level.INFO, () -> "Created Pod: " + cloudName + " " + namespace + "/" + podName);
                listener.getLogger().printf("Created Pod: %s %s/%s%n", cloudName, namespace, podName);
                Metrics.metricRegistry().counter("kubernetes.cloud.pods.created").inc();
                node.getRunListener().getLogger().printf("Created Pod: %s %s/%s%n", cloudName, namespace, podName);
            } else {
                LOGGER.log(Level.INFO, () -> "Pod already exists: " + cloudName + " " + namespace + "/" + podName);
                listener.getLogger().printf("Pod already exists: %s %s/%s%n", cloudName, namespace, podName);
            }
            kubernetesComputer.setLaunching(true);
            ObjectMeta podMetadata = pod.getMetadata();
            template.getWorkspaceVolume().createVolume(client, podMetadata);
            template.getVolumes().forEach(volume -> volume.createVolume(client, podMetadata));
            ((PodResource)((NonNamespaceOperation)client.pods().inNamespace(namespace)).withName(podName)).waitUntilReady((long)template.getSlaveConnectTimeout(), TimeUnit.SECONDS);
            LOGGER.log(Level.INFO, () -> "Pod is running: " + cloudName + " " + namespace + "/" + podName);
            int waitForSlaveToConnect = template.getSlaveConnectTimeout();
            SlaveComputer slaveComputer = null;
            String status = null;
            List containerStatuses = null;
            long lastReportTimestamp = System.currentTimeMillis();
            for (waitedForSlave = 0; waitedForSlave < waitForSlaveToConnect; ++waitedForSlave) {
                slaveComputer = node.getComputer();
                if (slaveComputer == null) {
                    Metrics.metricRegistry().counter("kubernetes.cloud.pods.launch.failed").inc();
                    throw new IllegalStateException("Node was deleted, computer is null");
                }
                if (slaveComputer.isOnline()) break;
                pod = (Pod)((PodResource)((NonNamespaceOperation)client.pods().inNamespace(namespace)).withName(podName)).get();
                if (pod == null) {
                    Metrics.metricRegistry().counter("kubernetes.cloud.pods.launch.failed").inc();
                    throw new IllegalStateException("Pod no longer exists: " + podName);
                }
                status = pod.getStatus().getPhase();
                if (POD_TERMINATED_STATES.contains(status)) {
                    Metrics.metricRegistry().counter("kubernetes.cloud.pods.launch.failed").inc();
                    Metrics.metricRegistry().counter(MetricNames.metricNameForPodStatus(status)).inc();
                    this.logLastLines(containerStatuses, podName, namespace, node, null, client);
                    throw new IllegalStateException("Pod '" + podName + "' is terminated. Status: " + status);
                }
                containerStatuses = pod.getStatus().getContainerStatuses();
                ArrayList<ContainerStatus> terminatedContainers = new ArrayList<ContainerStatus>();
                for (ContainerStatus info : containerStatuses) {
                    if (info == null || info.getState().getTerminated() == null) continue;
                    LOGGER.log(Level.INFO, "Container is terminated {0} [{2}]: {1}", new Object[]{podName, info.getState().getTerminated(), info.getName()});
                    listener.getLogger().printf("Container is terminated %1$s [%3$s]: %2$s%n", podName, info.getState().getTerminated(), info.getName());
                    Metrics.metricRegistry().counter("kubernetes.cloud.pods.launch.failed").inc();
                    terminatedContainers.add(info);
                }
                this.checkTerminatedContainers(terminatedContainers, podName, namespace, node, client);
                if (lastReportTimestamp + REPORT_INTERVAL < System.currentTimeMillis()) {
                    LOGGER.log(Level.INFO, "Waiting for agent to connect ({1}/{2}): {0}", new Object[]{podName, waitedForSlave, waitForSlaveToConnect});
                    listener.getLogger().printf("Waiting for agent to connect (%2$s/%3$s): %1$s%n", podName, waitedForSlave, waitForSlaveToConnect);
                    lastReportTimestamp = System.currentTimeMillis();
                }
                Thread.sleep(1000L);
            }
            if (slaveComputer == null || slaveComputer.isOffline()) {
                Metrics.metricRegistry().counter("kubernetes.cloud.pods.launch.failed").inc();
                Metrics.metricRegistry().counter("kubernetes.cloud.pods.launch.failed.timeout").inc();
                this.logLastLines(containerStatuses, podName, namespace, node, null, client);
                throw new IllegalStateException("Agent is not connected after " + waitedForSlave + " seconds, status: " + status);
            }
            computer.setAcceptingTasks(true);
            this.launched = true;
            try {
                node.save();
            }
            catch (IOException e) {
                LOGGER.log(Level.WARNING, "Could not save() agent: " + e.getMessage(), e);
            }
            Metrics.metricRegistry().counter("kubernetes.cloud.pods.launched").inc();
        }
        catch (Throwable ex) {
            this.setProblem(ex);
            Functions.printStackTrace((Throwable)ex, (PrintWriter)node.getRunListener().error("Failed to launch " + node.getPodName()));
            LOGGER.log(Level.WARNING, String.format("Error in provisioning; agent=%s, template=%s", new Object[]{node, node.getTemplateId()}), ex);
            LOGGER.log(Level.FINER, "Removing Jenkins node: {0}", node.getNodeName());
            KubernetesLauncher.terminateOrLog(node);
            throw new RuntimeException(ex);
        }
    }

    private static void terminateOrLog(KubernetesSlave node) {
        try {
            node.terminate();
        }
        catch (IOException | InterruptedException e) {
            LOGGER.log(Level.WARNING, "Unable to remove Jenkins node", e);
        }
    }

    private void checkTerminatedContainers(List<ContainerStatus> terminatedContainers, String podId, String namespace, KubernetesSlave slave, KubernetesClient client) {
        if (!terminatedContainers.isEmpty()) {
            Map<String, Integer> errors = terminatedContainers.stream().collect(Collectors.toMap(ContainerStatus::getName, info -> info.getState().getTerminated().getExitCode()));
            this.logLastLines(terminatedContainers, podId, namespace, slave, errors, client);
            throw new IllegalStateException("Containers are terminated with exit codes: " + String.valueOf(errors));
        }
    }

    private void logLastLines(@CheckForNull List<ContainerStatus> containers, String podId, String namespace, KubernetesSlave slave, Map<String, Integer> errors, KubernetesClient client) {
        if (containers != null) {
            for (ContainerStatus containerStatus : containers) {
                String containerName = containerStatus.getName();
                String log = ((ContainerResource)((PodResource)((NonNamespaceOperation)client.pods().inNamespace(namespace)).withName(podId)).inContainer((Object)containerStatus.getName())).tailingLines(30).getLog();
                if (StringUtils.isBlank((String)log)) continue;
                String msg = errors != null ? String.format(" exited with error %s", errors.get(containerName)) : "";
                LOGGER.log(Level.SEVERE, "Error in provisioning; agent={0}, template={1}. Container {2}{3}. Logs: {4}", new Object[]{slave, slave.getTemplateOrNull(), containerName, msg, log});
            }
        }
    }

    @CheckForNull
    public Throwable getProblem() {
        return this.problem;
    }

    public void setProblem(@CheckForNull Throwable problem) {
        this.problem = problem;
    }

    public Descriptor<ComputerLauncher> getDescriptor() {
        return new DescriptorImpl();
    }

    private static class DescriptorImpl
    extends Descriptor<ComputerLauncher> {
        private DescriptorImpl() {
        }
    }
}

