/*
 * Decompiled with CFR 0.152.
 */
package org.csanchez.jenkins.plugins.kubernetes.pod.retention;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import edu.umd.cs.findbugs.annotations.CheckForNull;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import hudson.Extension;
import hudson.ExtensionList;
import hudson.ExtensionPoint;
import hudson.XmlFile;
import hudson.model.Computer;
import hudson.model.Node;
import hudson.model.Saveable;
import hudson.model.TaskListener;
import hudson.model.listeners.ItemListener;
import hudson.model.listeners.SaveableListener;
import hudson.slaves.ComputerListener;
import hudson.slaves.OfflineCause;
import hudson.slaves.SlaveComputer;
import io.fabric8.kubernetes.api.model.ContainerStateTerminated;
import io.fabric8.kubernetes.api.model.ContainerStateWaiting;
import io.fabric8.kubernetes.api.model.ContainerStatus;
import io.fabric8.kubernetes.api.model.Pod;
import io.fabric8.kubernetes.api.model.PodCondition;
import io.fabric8.kubernetes.client.KubernetesClient;
import io.fabric8.kubernetes.client.Watch;
import io.fabric8.kubernetes.client.Watcher;
import io.fabric8.kubernetes.client.WatcherException;
import io.fabric8.kubernetes.client.dsl.NonNamespaceOperation;
import io.fabric8.kubernetes.client.dsl.PodResource;
import java.io.IOException;
import java.lang.invoke.CallSite;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;
import jenkins.model.Jenkins;
import jenkins.util.Listeners;
import jenkins.util.SystemProperties;
import jenkins.util.Timer;
import org.csanchez.jenkins.plugins.kubernetes.KubernetesClientProvider;
import org.csanchez.jenkins.plugins.kubernetes.KubernetesCloud;
import org.csanchez.jenkins.plugins.kubernetes.KubernetesComputer;
import org.csanchez.jenkins.plugins.kubernetes.KubernetesSlave;
import org.csanchez.jenkins.plugins.kubernetes.PodUtils;
import org.csanchez.jenkins.plugins.kubernetes.pod.retention.Messages;
import org.csanchez.jenkins.plugins.kubernetes.pod.retention.PodOfflineCause;
import org.jenkinsci.plugins.kubernetes.auth.KubernetesAuthException;

@Extension
public class Reaper
extends ComputerListener {
    private static final Logger LOGGER = Logger.getLogger(Reaper.class.getName());
    private final AtomicBoolean activated = new AtomicBoolean();
    private final Map<String, CloudPodWatcher> watchers = new ConcurrentHashMap<String, CloudPodWatcher>();
    private final LoadingCache<String, Set<String>> terminationReasons = Caffeine.newBuilder().expireAfterAccess(1L, TimeUnit.DAYS).build(k -> new ConcurrentSkipListSet());

    public static Reaper getInstance() {
        return (Reaper)((Object)ExtensionList.lookupSingleton(Reaper.class));
    }

    public void preLaunch(Computer c, TaskListener taskListener) throws IOException, InterruptedException {
        if (c instanceof KubernetesComputer) {
            Timer.get().schedule(this::maybeActivate, 10L, TimeUnit.SECONDS);
            KubernetesSlave node = (KubernetesSlave)((KubernetesComputer)c).getNode();
            if (node != null && !this.isWatchingCloud(node.getCloudName())) {
                try {
                    this.watchCloud(node.getKubernetesCloud());
                }
                catch (IllegalStateException ise) {
                    LOGGER.log(Level.WARNING, ise, () -> "kubernetes cloud not found: " + node.getCloudName());
                }
            }
        }
    }

    public void maybeActivate() {
        if (this.activated.compareAndSet(false, true)) {
            this.activate();
        }
    }

    private void activate() {
        LOGGER.fine("Activating reaper");
        this.reapAgents();
        this.watchClouds();
    }

    private void reapAgents() {
        Jenkins jenkins = Jenkins.getInstanceOrNull();
        if (jenkins != null) {
            for (Node n : new ArrayList(jenkins.getNodes())) {
                KubernetesSlave ks;
                if (!(n instanceof KubernetesSlave) || (ks = (KubernetesSlave)n).getLauncher().isLaunchSupported()) continue;
                String ns = ks.getNamespace();
                String name = ks.getPodName();
                try {
                    KubernetesCloud cloud = ks.getKubernetesCloud();
                    if (((PodResource)((NonNamespaceOperation)cloud.connect().pods().inNamespace(ns)).withName(name)).get() == null) {
                        LOGGER.info(() -> ns + "/" + name + " seems to have been deleted, so removing corresponding Jenkins agent");
                        jenkins.removeNode((Node)ks);
                        continue;
                    }
                    LOGGER.fine(() -> ns + "/" + name + " still seems to exist, OK");
                }
                catch (IOException | RuntimeException | KubernetesAuthException x) {
                    LOGGER.log(Level.WARNING, x, () -> "failed to do initial reap check for " + ns + "/" + name);
                }
            }
        }
    }

    private void watchClouds() {
        Jenkins jenkins = Jenkins.getInstanceOrNull();
        if (jenkins != null) {
            HashSet<String> cloudNames = new HashSet<String>(this.watchers.keySet());
            for (KubernetesCloud kc : jenkins.clouds.getAll(KubernetesCloud.class)) {
                this.watchCloud(kc);
                cloudNames.remove(kc.name);
            }
            cloudNames.stream().map(this.watchers::get).filter(Objects::nonNull).forEach(cpw -> {
                LOGGER.info(() -> "stopping pod watcher for deleted kubernetes cloud " + cpw.cloudName);
                cpw.stop();
            });
        }
    }

    private void watchCloud(@NonNull KubernetesCloud kc) {
        CloudPodWatcher watcher = new CloudPodWatcher(kc);
        if (!this.isCloudPodWatcherActive(watcher)) {
            try {
                KubernetesClient client = kc.connect();
                watcher.watch = ((NonNamespaceOperation)client.pods().inNamespace(client.getNamespace())).watch((Watcher)watcher);
                CloudPodWatcher old = this.watchers.put(kc.name, watcher);
                if (old != null) {
                    old.stop();
                }
                LOGGER.info(() -> "set up watcher on " + kc.getDisplayName());
            }
            catch (IOException | RuntimeException | KubernetesAuthException x) {
                LOGGER.log(Level.WARNING, x, () -> "failed to set up watcher on " + kc.getDisplayName());
            }
        }
    }

    boolean isWatchingCloud(String name) {
        return this.watchers.get(name) != null;
    }

    public Map<String, ?> getWatchers() {
        return this.watchers;
    }

    private boolean isCloudPodWatcherActive(@NonNull CloudPodWatcher watcher) {
        CloudPodWatcher existing = this.watchers.get(watcher.cloudName);
        return existing != null && existing.clientValidity == watcher.clientValidity;
    }

    private static Optional<KubernetesSlave> resolveNode(@NonNull Jenkins jenkins, String namespace, String name) {
        return new ArrayList(jenkins.getNodes()).stream().filter(KubernetesSlave.class::isInstance).map(KubernetesSlave.class::cast).filter(ks -> Objects.equals(ks.getNamespace(), namespace) && Objects.equals(ks.getPodName(), name)).findFirst();
    }

    private void closeAllWatchers() {
        this.watchers.values().forEach(CloudPodWatcher::stop);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @NonNull
    public Set<String> terminationReasons(@NonNull String node) {
        LoadingCache<String, Set<String>> loadingCache = this.terminationReasons;
        synchronized (loadingCache) {
            return new HashSet<String>((Collection)this.terminationReasons.get((Object)node));
        }
    }

    private static void logAndCleanUp(KubernetesSlave node, Pod pod, Set<String> terminationReasons, String reason, String message, StringBuilder sb, TaskListener runListener, PodOfflineCause cause) throws IOException, InterruptedException {
        Optional<PodCondition> evictionCondition;
        ArrayList<CallSite> details = new ArrayList<CallSite>();
        if (reason != null) {
            details.add((CallSite)((Object)("Reason: " + reason)));
            terminationReasons.add(reason);
        }
        if (message != null) {
            details.add((CallSite)((Object)("Message: " + message)));
        }
        if (!details.isEmpty()) {
            sb.append(" ").append(String.join((CharSequence)", ", details)).append(".");
        }
        if ((evictionCondition = pod.getStatus().getConditions().stream().filter(c -> "EvictionByEvictionAPI".equals(c.getReason())).findFirst()).isPresent()) {
            sb.append(" Pod was evicted by the Kubernetes Eviction API.");
            terminationReasons.add(evictionCondition.get().getReason());
        }
        LOGGER.info(() -> String.valueOf(sb) + " Removing corresponding node " + node.getNodeName() + " from Jenkins.");
        runListener.getLogger().println(sb);
        Reaper.logLastLinesThenTerminateNode(node, pod, runListener);
        PodUtils.cancelQueueItemFor(pod, "PodFailure");
        Reaper.disconnectComputer(node, (OfflineCause)cause);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void logLastLinesThenTerminateNode(KubernetesSlave node, Pod pod, TaskListener runListener) throws IOException, InterruptedException {
        try {
            String lines = PodUtils.logLastLines(pod, node.getKubernetesCloud().connect());
            if (lines != null) {
                runListener.getLogger().print(lines);
                String ns = pod.getMetadata().getNamespace();
                String name = pod.getMetadata().getName();
                LOGGER.fine(() -> ns + "/" + name + " log:\n" + lines);
            }
        }
        catch (KubernetesAuthException e) {
            LOGGER.log(Level.FINE, e, () -> "Unable to get logs after pod failed event");
        }
        finally {
            node.terminate();
        }
    }

    private static void disconnectComputer(KubernetesSlave node, OfflineCause cause) {
        SlaveComputer computer = node.getComputer();
        if (computer != null) {
            computer.disconnect(cause);
        }
    }

    private class CloudPodWatcher
    implements Watcher<Pod> {
        private final String cloudName;
        private final int clientValidity;
        @CheckForNull
        private Watch watch;

        CloudPodWatcher(KubernetesCloud cloud) {
            this.cloudName = cloud.name;
            this.clientValidity = KubernetesClientProvider.getValidity(cloud);
        }

        public void eventReceived(Watcher.Action action, Pod pod) {
            String name;
            if (action == Watcher.Action.BOOKMARK) {
                return;
            }
            if (action == Watcher.Action.ERROR && pod == null) {
                return;
            }
            Jenkins jenkins = Jenkins.getInstanceOrNull();
            if (jenkins == null) {
                return;
            }
            String ns = pod.getMetadata().getNamespace();
            Optional<KubernetesSlave> optionalNode = Reaper.resolveNode(jenkins, ns, name = pod.getMetadata().getName());
            if (!optionalNode.isPresent()) {
                return;
            }
            Listeners.notify(Listener.class, (boolean)true, listener -> {
                try {
                    Set<String> terminationReasons = (Set<String>)Reaper.this.terminationReasons.get((Object)((KubernetesSlave)((Object)((Object)optionalNode.get()))).getNodeName());
                    listener.onEvent(action, (KubernetesSlave)((Object)((Object)optionalNode.get())), pod, terminationReasons != null ? terminationReasons : Collections.emptySet());
                }
                catch (Exception x) {
                    LOGGER.log(Level.WARNING, "Listener " + String.valueOf(listener) + " failed for " + ns + "/" + name, x);
                }
            });
        }

        void stop() {
            if (this.watch != null) {
                LOGGER.info("Stopping watch for kubernetes cloud " + this.cloudName);
                this.watch.close();
            }
        }

        public void onClose() {
            LOGGER.fine(() -> this.cloudName + " watcher closed");
            Reaper.this.watchers.remove(this.cloudName, this);
        }

        public void onClose(WatcherException e) {
            LOGGER.log(Level.WARNING, (Throwable)e, () -> this.cloudName + " watcher closed with exception");
            Reaper.this.watchers.remove(this.cloudName, this);
        }
    }

    @Extension
    public static class ReaperSaveableListener
    extends SaveableListener {
        public void onChange(Saveable o, XmlFile file) {
            if (o instanceof Jenkins) {
                Reaper reaper = Reaper.getInstance();
                if (reaper.activated.get()) {
                    Reaper.getInstance().watchClouds();
                }
            }
        }
    }

    @Extension
    public static class TerminateAgentOnImagePullBackOff
    implements Listener {
        @SuppressFBWarnings(value={"MS_SHOULD_BE_FINAL"}, justification="Allow tests or groovy console to change the value")
        public static long BACKOFF_EVENTS_LIMIT = SystemProperties.getInteger((String)(Reaper.class.getName() + ".backoffEventsLimit"), (Integer)3).intValue();
        public static final String IMAGE_PULL_BACK_OFF = "ImagePullBackOff";
        private Cache<String, Integer> ttlCache = Caffeine.newBuilder().expireAfterWrite(15L, TimeUnit.MINUTES).build();

        @Override
        public void onEvent(@NonNull Watcher.Action action, @NonNull KubernetesSlave node, @NonNull Pod pod, @NonNull Set<String> terminationReasons) throws IOException, InterruptedException {
            if (action != Watcher.Action.MODIFIED) {
                return;
            }
            List<ContainerStatus> backOffContainers = PodUtils.getContainers(pod, cs -> {
                ContainerStateWaiting waiting = cs.getState().getWaiting();
                return waiting != null && waiting.getMessage() != null && waiting.getMessage().contains("Back-off pulling image");
            });
            if (!backOffContainers.isEmpty()) {
                ArrayList images = new ArrayList();
                backOffContainers.forEach(cs -> images.add(cs.getImage()));
                String podUid = pod.getMetadata().getUid();
                Integer backOffNumber = (Integer)this.ttlCache.get((Object)podUid, k -> 0);
                backOffNumber = backOffNumber + 1;
                this.ttlCache.put((Object)podUid, (Object)backOffNumber);
                if ((long)backOffNumber.intValue() >= BACKOFF_EVENTS_LIMIT) {
                    String imagesString = String.join((CharSequence)",", images);
                    node.getRunListener().error("Unable to pull container image \"" + imagesString + "\". Check if image tag name is spelled correctly.");
                    terminationReasons.add(IMAGE_PULL_BACK_OFF);
                    PodUtils.cancelQueueItemFor(pod, IMAGE_PULL_BACK_OFF);
                    node.terminate();
                    Reaper.disconnectComputer(node, (OfflineCause)new PodOfflineCause(Messages._PodOfflineCause_ImagePullBackoff(IMAGE_PULL_BACK_OFF, images)));
                } else {
                    node.getRunListener().error("Image pull backoff detected, waiting for image to be available. Will wait for " + (BACKOFF_EVENTS_LIMIT - (long)backOffNumber.intValue()) + " more events before terminating the node.");
                }
            }
        }
    }

    @Extension
    public static class TerminateAgentOnPodFailed
    implements Listener {
        @Override
        public void onEvent(@NonNull Watcher.Action action, @NonNull KubernetesSlave node, @NonNull Pod pod, @NonNull Set<String> terminationReasons) throws IOException, InterruptedException {
            if (action != Watcher.Action.MODIFIED) {
                return;
            }
            if ("Failed".equals(pod.getStatus().getPhase())) {
                String reason = pod.getStatus().getReason();
                String message = pod.getStatus().getMessage();
                Reaper.logAndCleanUp(node, pod, terminationReasons, reason, message, new StringBuilder().append(pod.getMetadata().getNamespace()).append("/").append(pod.getMetadata().getName()).append(" Pod just failed."), node.getRunListener(), new PodOfflineCause(Messages._PodOfflineCause_PodFailed(reason, message)));
            }
        }
    }

    @Extension
    public static class TerminateAgentOnContainerTerminated
    implements Listener {
        @Override
        public void onEvent(@NonNull Watcher.Action action, @NonNull KubernetesSlave node, @NonNull Pod pod, @NonNull Set<String> terminationReasons) throws IOException, InterruptedException {
            if (action != Watcher.Action.MODIFIED) {
                return;
            }
            List<ContainerStatus> terminatedContainers = PodUtils.getTerminatedContainers(pod);
            if (!terminatedContainers.isEmpty()) {
                ArrayList containers = new ArrayList();
                terminatedContainers.forEach(c -> {
                    ContainerStateTerminated t = c.getState().getTerminated();
                    String containerName = c.getName();
                    containers.add(containerName);
                    String reason = t.getReason();
                    if (reason != null) {
                        terminationReasons.add(reason);
                    }
                });
                String reason = pod.getStatus().getReason();
                String message = pod.getStatus().getMessage();
                StringBuilder sb = new StringBuilder().append(pod.getMetadata().getNamespace()).append("/").append(pod.getMetadata().getName());
                if (containers.size() > 1) {
                    sb.append(" Containers ").append(String.join((CharSequence)",", containers)).append(" were terminated.");
                } else {
                    sb.append(" Container ").append(String.join((CharSequence)",", containers)).append(" was terminated.");
                }
                Reaper.logAndCleanUp(node, pod, terminationReasons, reason, message, sb, node.getRunListener(), new PodOfflineCause(Messages._PodOfflineCause_ContainerFailed("ContainerError", containers)));
            }
        }
    }

    @Extension
    public static class RemoveAgentOnPodDeleted
    implements Listener {
        @Override
        public void onEvent(@NonNull Watcher.Action action, @NonNull KubernetesSlave node, @NonNull Pod pod, @NonNull Set<String> terminationReasons) throws IOException {
            if (action != Watcher.Action.DELETED) {
                return;
            }
            String ns = pod.getMetadata().getNamespace();
            String name = pod.getMetadata().getName();
            LOGGER.info(() -> ns + "/" + name + " was just deleted, so removing corresponding Jenkins agent");
            node.getRunListener().getLogger().printf("Pod %s/%s was just deleted%n", ns, name);
            Jenkins.get().removeNode((Node)node);
            Reaper.disconnectComputer(node, (OfflineCause)new PodOfflineCause(Messages._PodOfflineCause_PodDeleted()));
        }
    }

    public static interface Listener
    extends ExtensionPoint {
        public void onEvent(@NonNull Watcher.Action var1, @NonNull KubernetesSlave var2, @NonNull Pod var3, @NonNull Set<String> var4) throws IOException, InterruptedException;
    }

    @Extension
    public static class ReaperShutdownListener
    extends ItemListener {
        public void onBeforeShutdown() {
            Reaper.getInstance().closeAllWatchers();
        }
    }
}

