/*
 * Decompiled with CFR 0.152.
 */
package org.csanchez.jenkins.plugins.kubernetes;

import edu.umd.cs.findbugs.annotations.CheckForNull;
import edu.umd.cs.findbugs.annotations.NonNull;
import hudson.Util;
import hudson.model.Label;
import hudson.model.Queue;
import io.fabric8.kubernetes.api.model.ContainerStatus;
import io.fabric8.kubernetes.api.model.ObjectMeta;
import io.fabric8.kubernetes.api.model.Pod;
import io.fabric8.kubernetes.api.model.PodStatus;
import io.fabric8.kubernetes.client.KubernetesClient;
import io.fabric8.kubernetes.client.KubernetesClientException;
import io.fabric8.kubernetes.client.dsl.ContainerResource;
import io.fabric8.kubernetes.client.dsl.NonNamespaceOperation;
import io.fabric8.kubernetes.client.dsl.PodResource;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Predicate;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import jenkins.model.Jenkins;
import org.apache.commons.lang.RandomStringUtils;
import org.apache.commons.lang.StringUtils;
import org.csanchez.jenkins.plugins.kubernetes.PodTemplateUtils;

public final class PodUtils {
    private static final Logger LOGGER = Logger.getLogger(PodUtils.class.getName());
    private static final Pattern NAME_PATTERN = Pattern.compile("[a-z0-9]([-a-z0-9]*[a-z0-9])?(\\.[a-z0-9]([-a-z0-9]*[a-z0-9])?)*");
    public static final Predicate<ContainerStatus> CONTAINER_IS_TERMINATED = cs -> cs.getState().getTerminated() != null;
    public static final Predicate<ContainerStatus> CONTAINER_IS_WAITING = cs -> cs.getState().getWaiting() != null;

    private PodUtils() {
    }

    @NonNull
    public static List<ContainerStatus> getTerminatedContainers(Pod pod) {
        return PodUtils.getContainers(pod, CONTAINER_IS_TERMINATED);
    }

    public static List<ContainerStatus> getWaitingContainers(Pod pod) {
        return PodUtils.getContainers(pod, CONTAINER_IS_WAITING);
    }

    @NonNull
    public static List<ContainerStatus> getContainerStatus(@NonNull Pod pod) {
        PodStatus podStatus = pod.getStatus();
        if (podStatus == null) {
            return Collections.emptyList();
        }
        return podStatus.getContainerStatuses();
    }

    public static List<ContainerStatus> getContainers(@NonNull Pod pod, @NonNull Predicate<ContainerStatus> predicate) {
        return PodUtils.getContainerStatus(pod).stream().filter(predicate).collect(Collectors.toList());
    }

    public static void cancelQueueItemFor(Pod pod, String reason) {
        ObjectMeta metadata = pod.getMetadata();
        if (metadata == null) {
            return;
        }
        String podName = metadata.getName();
        String podNamespace = metadata.getNamespace();
        String podDisplayName = podNamespace + "/" + podName;
        Map annotations = metadata.getAnnotations();
        if (annotations == null) {
            LOGGER.log(Level.FINE, () -> "Pod " + podDisplayName + " .metadata.annotations is null");
            return;
        }
        String runUrl = (String)annotations.get("runUrl");
        if (runUrl == null) {
            LOGGER.log(Level.FINE, () -> "Pod " + podDisplayName + " .metadata.annotations.runUrl is null");
            return;
        }
        Map labels = metadata.getLabels();
        if (labels == null) {
            LOGGER.log(Level.FINE, () -> "Pod " + podDisplayName + " .metadata.labels is null");
            return;
        }
        PodUtils.cancelQueueItemFor(runUrl, (String)labels.get("jenkins/label"), reason, podDisplayName);
    }

    public static void cancelQueueItemFor(@NonNull String runUrl, @NonNull String label, @CheckForNull String reason, @CheckForNull String podDisplayName) {
        Queue queue = Jenkins.get().getQueue();
        Arrays.stream(queue.getItems()).filter(item -> Objects.equals(item.getTask().getUrl(), runUrl)).filter(item -> Optional.ofNullable(item.getAssignedLabel()).map(Label::getName).map(name -> PodTemplateUtils.sanitizeLabel(name).equals(label)).orElse(false)).findFirst().ifPresentOrElse(item -> {
            LOGGER.log(Level.FINE, () -> "Cancelling queue item: \"" + item.task.getDisplayName() + "\"\n" + (String)(!StringUtils.isBlank((String)reason) ? "due to " + reason : ""));
            queue.cancel(item);
        }, () -> {
            if (podDisplayName != null) {
                LOGGER.log(Level.FINE, () -> "No queue item found for pod " + podDisplayName);
            }
        });
    }

    @CheckForNull
    public static String logLastLines(@NonNull Pod pod, @NonNull KubernetesClient client) {
        PodStatus status = pod.getStatus();
        ObjectMeta metadata = pod.getMetadata();
        if (status == null || metadata == null) {
            return null;
        }
        String podName = metadata.getName();
        String namespace = metadata.getNamespace();
        List containers = status.getContainerStatuses();
        StringBuilder sb = new StringBuilder();
        if (containers != null) {
            for (ContainerStatus containerStatus : containers) {
                sb.append("\n");
                sb.append("- ");
                sb.append(containerStatus.getName());
                if (containerStatus.getState().getTerminated() != null) {
                    sb.append(" -- terminated (");
                    sb.append(containerStatus.getState().getTerminated().getExitCode());
                    sb.append(")");
                }
                if (containerStatus.getState().getRunning() != null) {
                    sb.append(" -- running");
                }
                if (containerStatus.getState().getWaiting() != null) {
                    sb.append(" -- waiting");
                }
                sb.append("\n");
                try {
                    String log = ((ContainerResource)((PodResource)((NonNamespaceOperation)client.pods().inNamespace(namespace)).withName(podName)).inContainer((Object)containerStatus.getName())).tailingLines(30).getLog();
                    sb.append("-----Logs-------------\n");
                    sb.append(log);
                    sb.append("\n");
                }
                catch (KubernetesClientException e) {
                    LOGGER.log(Level.FINE, e, () -> namespace + "/" + podName + " Unable to retrieve container logs as the pod is already gone");
                }
            }
        }
        return Util.fixEmpty((String)sb.toString());
    }

    @NonNull
    public static String generateRandomSuffix() {
        return RandomStringUtils.random((int)5, (String)"bcdfghjklmnpqrstvwxz0123456789");
    }

    @NonNull
    public static String createNameWithRandomSuffix(@NonNull String name) {
        String suffix = PodUtils.generateRandomSuffix();
        name = name.replaceAll("[ _]", "-").toLowerCase(Locale.getDefault());
        name = name.substring(0, Math.min(name.length(), 62 - suffix.length()));
        return String.join((CharSequence)"-", name, suffix);
    }

    public static boolean isValidName(@NonNull String name) {
        return NAME_PATTERN.matcher(name).matches();
    }
}

