/*
 * Decompiled with CFR 0.152.
 */
package org.marvelution.jji.tunnel;

import com.fasterxml.jackson.databind.ObjectMapper;
import hudson.Extension;
import hudson.Plugin;
import hudson.PluginWrapper;
import hudson.XmlFile;
import hudson.init.InitMilestone;
import hudson.init.Initializer;
import hudson.model.Saveable;
import hudson.model.listeners.SaveableListener;
import hudson.util.PluginServletFilter;
import java.io.IOException;
import java.net.URI;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.Nullable;
import javax.inject.Inject;
import javax.inject.Provider;
import javax.servlet.Filter;
import javax.servlet.ServletException;
import jenkins.model.Jenkins;
import jenkins.model.JenkinsLocationConfiguration;
import okhttp3.OkHttpClient;
import okhttp3.Response;
import okhttp3.ResponseBody;
import org.marvelution.jji.Messages;
import org.marvelution.jji.configuration.JiraSite;
import org.marvelution.jji.configuration.JiraSitesConfiguration;
import org.marvelution.jji.tunnel.NgrokConnector;
import org.marvelution.jji.tunnel.TunnelAuthenticationFilter;
import org.marvelution.jji.tunnel.TunnelDetails;
import org.springframework.security.access.AccessDeniedException;

@Extension
public class TunnelManager
extends SaveableListener {
    private static final Logger LOGGER = Logger.getLogger(TunnelManager.class.getName());
    private static final String UNKNOWN = "unknown";
    private final Map<String, NgrokConnector> tunnels = new ConcurrentHashMap<String, NgrokConnector>();
    private ClassLoader tunnelClassLoader;
    private boolean loadedTunnelFilter = false;
    private String forwardTo;
    private Provider<OkHttpClient> httpClient;
    private ObjectMapper objectMapper;

    public void onChange(Saveable o, XmlFile file) {
        String newForwardTo;
        if (o instanceof JiraSitesConfiguration) {
            LOGGER.info("Jira Sites configuration change detected, connecting new tunnels and disconnecting old once.");
            Set<JiraSite> sites = ((JiraSitesConfiguration)o).getSites();
            sites.forEach(this::connectTunnelIfNeeded);
            Iterator<Map.Entry<String, NgrokConnector>> it = this.tunnels.entrySet().iterator();
            while (it.hasNext()) {
                Map.Entry<String, NgrokConnector> entry = it.next();
                if (!sites.stream().noneMatch(site -> Objects.equals(site.getIdentifier(), entry.getKey()))) continue;
                entry.getValue().close();
                it.remove();
            }
        } else if (o instanceof JenkinsLocationConfiguration && !Objects.equals(this.forwardTo, newForwardTo = this.getForwardTo((JenkinsLocationConfiguration)o))) {
            LOGGER.info("Jenkins url change detected, reconnecting existing tunnels.");
            this.forwardTo = newForwardTo;
            this.tunnels.values().forEach(NgrokConnector::reconnect);
        }
    }

    public void verifyTunnelToken(String tunnelId, String tunnelToken) {
        NgrokConnector connector = this.tunnels.get(tunnelId);
        if (connector == null) {
            throw new AccessDeniedException("unknown tunnel id");
        }
        connector.verifyTunnelToken(tunnelId, tunnelToken);
    }

    public String getForwardTo() {
        if (this.forwardTo == null) {
            this.forwardTo = this.getForwardTo(JenkinsLocationConfiguration.get());
        }
        return this.forwardTo;
    }

    private String getForwardTo(JenkinsLocationConfiguration configuration) {
        URI rootUrl = URI.create(Optional.ofNullable(configuration.getUrl()).orElse("http://localhost:8080/"));
        String forwardTo = rootUrl.getHost() + ":";
        forwardTo = rootUrl.getPort() != -1 ? forwardTo + rootUrl.getPort() : ("https".equals(rootUrl.getScheme()) ? forwardTo + "443" : forwardTo + "80");
        return forwardTo;
    }

    private static String normalizeOs(String value) {
        if ((value = TunnelManager.normalize(value)).startsWith("linux")) {
            return "linux";
        }
        if (value.startsWith("mac") || value.startsWith("osx")) {
            return "osx";
        }
        if (value.startsWith("windows")) {
            return "windows";
        }
        return UNKNOWN;
    }

    private static String normalizeArch(String value) {
        if ((value = TunnelManager.normalize(value)).matches("^(x8664|amd64|ia32e|em64t|x64)$")) {
            return "x86_64";
        }
        if (value.matches("^(x8632|x86|i[3-6]86|ia32|x32)$")) {
            return "x86_32";
        }
        if ("aarch64".equals(value)) {
            return "aarch_64";
        }
        return UNKNOWN;
    }

    public ClassLoader getTunnelClassLoader() {
        return this.tunnelClassLoader;
    }

    private static String normalize(String value) {
        if (value == null) {
            return "";
        }
        return value.toLowerCase(Locale.US).replaceAll("[^a-z0-9]+", "");
    }

    @Initializer(after=InitMilestone.PLUGINS_PREPARED)
    public void loadNgrokNativeLibrary() {
        Plugin plugin = Jenkins.get().getPlugin("jira-integration");
        if (plugin != null) {
            PluginWrapper wrapper = plugin.getWrapper();
            String os = TunnelManager.normalizeOs(System.getProperty("os.name"));
            String arch = TunnelManager.normalizeArch(System.getProperty("os.arch"));
            String libraryFile = "WEB-INF/ngrok/ngrok-java-native-" + os + "-" + arch + ".jar";
            try {
                URL library = new URL(wrapper.baseResourceURL, libraryFile);
                LOGGER.info("Created classloader to support tunneling using " + String.valueOf(library));
                this.tunnelClassLoader = new URLClassLoader(new URL[]{library}, wrapper.classLoader);
            }
            catch (Exception e) {
                LOGGER.log(Level.WARNING, "Disabling tunneling, unable to add ngrok-java-native library to plugin classpath: " + libraryFile, e);
            }
        } else {
            throw new IllegalStateException("Unable to locate plugin jira-integration");
        }
    }

    @Initializer(after=InitMilestone.SYSTEM_CONFIG_ADAPTED)
    public void addTunnelAuthenticationFilter() {
        if (this.tunnelClassLoader != null) {
            LOGGER.log(Level.INFO, "Adding tunnel authentication filter");
            TunnelAuthenticationFilter filter = new TunnelAuthenticationFilter(this);
            if (!PluginServletFilter.hasFilter((Filter)filter)) {
                try {
                    PluginServletFilter.addFilter((Filter)filter);
                    this.loadedTunnelFilter = true;
                }
                catch (ServletException e) {
                    LOGGER.log(Level.WARNING, "Failed to set up tunnel authentication servlet filter", e);
                }
            }
        } else {
            LOGGER.log(Level.WARNING, "Skipping set up of tunnel authentication servlet filter, required libraries are not loaded.");
        }
    }

    @Initializer(after=InitMilestone.JOB_CONFIG_ADAPTED)
    public void connectRequiredTunnels() {
        if (this.tunnelClassLoader != null && this.loadedTunnelFilter) {
            LOGGER.log(Level.INFO, "Connecting any required tunnels");
            JiraSitesConfiguration.get().getSites().forEach(this::connectTunnelIfNeeded);
            Runtime.getRuntime().addShutdownHook(new Thread(this::stopTunneling));
        } else {
            LOGGER.log(Level.WARNING, "Not connecting any required tunnels, required libraries are not loaded.");
        }
    }

    public void disconnectTunnelForSite(JiraSite site) {
        NgrokConnector connector = this.tunnels.remove(site.getIdentifier());
        if (connector != null) {
            connector.close();
        }
    }

    public void connectTunnelIfNeeded(JiraSite site) {
        if (site.isTunneled()) {
            if (this.tunnelClassLoader != null && this.loadedTunnelFilter) {
                this.tunnels.computeIfAbsent(site.getIdentifier(), id -> {
                    try (Response response = ((OkHttpClient)this.httpClient.get()).newCall(site.createGetTunnelDetailsRequest()).execute();
                         ResponseBody body = response.body();){
                        if (response.isSuccessful() && body != null) {
                            TunnelDetails details = (TunnelDetails)this.objectMapper.readValue(body.byteStream(), TunnelDetails.class);
                            NgrokConnector connector = new NgrokConnector(this, site, details);
                            connector.connect();
                            NgrokConnector ngrokConnector = connector;
                            return ngrokConnector;
                        }
                        LOGGER.severe(String.format("Failed to load tunnel details for %s [%d]", site, response.code()));
                        return null;
                    }
                    catch (IOException e) {
                        LOGGER.log(Level.SEVERE, String.format("Failed to load tunnel details for %s", site), e);
                    }
                    return null;
                });
            } else if (this.tunnelClassLoader == null) {
                LOGGER.log(Level.WARNING, "Cannot connect tunnel for " + String.valueOf(site) + ", required libraries are not loaded.");
            } else {
                LOGGER.log(Level.WARNING, "Cannot connect tunnel for " + String.valueOf(site) + ", required security filter is not loaded.");
            }
        }
    }

    public void refreshTunnel(JiraSite site) {
        this.disconnectTunnelForSite(site);
        this.connectTunnelIfNeeded(site);
    }

    @Nullable
    public String getSiteConnectionError(JiraSite site) {
        if (site.isTunneled()) {
            NgrokConnector connector = this.tunnels.get(site.getIdentifier());
            if (connector == null) {
                return Messages.site_tunnel_not_connected();
            }
            return Optional.ofNullable(connector.getConnectException()).map(error -> {
                if (this.isUnsatisfiedLinkError((Throwable)error)) {
                    return Messages.site_tunnel_unsatisfied_link_error();
                }
                return error.getMessage();
            }).orElse(null);
        }
        return null;
    }

    private boolean isUnsatisfiedLinkError(Throwable error) {
        if (error instanceof UnsatisfiedLinkError) {
            return true;
        }
        if (error.getCause() != null) {
            return this.isUnsatisfiedLinkError(error.getCause());
        }
        return false;
    }

    private void stopTunneling() {
        LOGGER.info("Disconnecting and Closing all tunnels");
        this.tunnels.forEach((id, connector) -> connector.close());
        this.tunnels.clear();
    }

    @Inject
    public void setHttpClient(Provider<OkHttpClient> httpClient) {
        this.httpClient = httpClient;
    }

    @Inject
    public void setObjectMapper(ObjectMapper objectMapper) {
        this.objectMapper = objectMapper;
    }
}

