/*
 * Decompiled with CFR 0.152.
 */
package org.jenkinsci.plugins.github_branch_source;

import edu.umd.cs.findbugs.annotations.NonNull;
import hudson.Util;
import hudson.model.TaskListener;
import hudson.util.LogTaskListener;
import java.io.IOException;
import java.util.List;
import java.util.Objects;
import java.util.Random;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.commons.lang3.StringUtils;
import org.jenkinsci.plugins.github_branch_source.Endpoint;
import org.jenkinsci.plugins.github_branch_source.GitHubConfiguration;
import org.jenkinsci.plugins.github_branch_source.GitHubConsoleNote;
import org.jenkinsci.plugins.github_branch_source.Messages;
import org.kohsuke.github.GHRateLimit;
import org.kohsuke.github.GitHub;
import org.kohsuke.github.RateLimitChecker;

/*
 * Uses 'sealed' constructs - enablewith --sealed true
 */
public enum ApiRateLimitChecker {
    ThrottleForNormalize(Messages.ApiRateLimitChecker_ThrottleForNormalize()){

        @Override
        public LocalChecker getChecker(@NonNull TaskListener listener, String apiUrl) {
            return new LocalChecker(listener){

                @Override
                long checkRateLimitImpl(@NonNull GHRateLimit.Record rateLimit, long count, long now) throws InterruptedException {
                    long expiration = now;
                    int buffer = ApiRateLimitChecker.calculateBuffer(rateLimit.getLimit());
                    if (rateLimit.getRemaining() < buffer) {
                        expiration = this.calculateExpirationWhenBufferExceeded(rateLimit, now, buffer);
                    } else {
                        int burst = ApiRateLimitChecker.calculateNormalizedBurst(rateLimit.getLimit());
                        long rateLimitResetMillis = rateLimit.getResetDate().getTime() - now;
                        double resetProgress = Math.max(0.0, (double)rateLimitResetMillis / MILLIS_PER_HOUR);
                        int ideal = (int)((double)(rateLimit.getLimit() - buffer - burst) * resetProgress) + buffer;
                        if (rateLimit.getRemaining() < ideal) {
                            double targetFraction = ((double)rateLimit.getRemaining() - (double)buffer * 1.1) / (double)(rateLimit.getLimit() - buffer - burst);
                            expiration = rateLimit.getResetDate().getTime() - Math.max(0L, (long)(targetFraction * MILLIS_PER_HOUR)) + (long)ENTROPY.nextInt(1000);
                            this.writeLog(String.format("Jenkins-Imposed API Limiter: Current quota for Github API usage has %d remaining (%d over budget). Next quota of %d in %s. Sleeping for %s.", rateLimit.getRemaining(), ideal - rateLimit.getRemaining(), rateLimit.getLimit(), Util.getTimeSpanString((long)rateLimitResetMillis), Util.getTimeSpanString((long)(1000L + expiration - now))));
                        }
                    }
                    if (expiration != now) {
                        this.writeLog("Jenkins is attempting to evenly distribute GitHub API requests. To configure a different rate limiting strategy, such as having Jenkins restrict GitHub API requests only when near or above the GitHub rate limit, go to \"GitHub API usage\" under \"Configure System\" in the Jenkins settings.");
                    }
                    return expiration;
                }
            };
        }
    }
    ,
    ThrottleOnOver(Messages.ApiRateLimitChecker_ThrottleOnOver()){

        @Override
        public LocalChecker getChecker(@NonNull TaskListener listener, String apiUrl) {
            return new LocalChecker(listener){

                @Override
                long checkRateLimitImpl(@NonNull GHRateLimit.Record rateLimit, long count, long now) throws InterruptedException {
                    int buffer = ApiRateLimitChecker.calculateBuffer(rateLimit.getLimit());
                    if (rateLimit.getRemaining() >= buffer) {
                        return now;
                    }
                    this.writeLog("Jenkins is restricting GitHub API requests only when near or above the rate limit. To configure a different rate limiting strategy, such as having Jenkins attempt to evenly distribute GitHub API requests, go to \"GitHub API usage\" under \"Configure System\" in the Jenkins settings.");
                    return this.calculateExpirationWhenBufferExceeded(rateLimit, now, buffer);
                }
            };
        }
    }
    ,
    NoThrottle(Messages.ApiRateLimitChecker_NoThrottle()){

        @Override
        public LocalChecker getChecker(@NonNull TaskListener listener, String apiUrl) {
            if ("https://api.github.com".equals(apiUrl)) {
                LocalChecker checker = ThrottleOnOver.getChecker(listener, apiUrl);
                checker.writeLog("GitHub throttling is disabled, which is not allowed for public GitHub usage, so ThrottleOnOver will be used instead. To configure a different rate limiting strategy, go to \"GitHub API usage\" under \"Configure System\" in the Jenkins settings.");
                return checker;
            }
            return new LocalChecker(listener){

                @Override
                long checkRateLimitImpl(@NonNull GHRateLimit.Record rateLimit, long count, long now) throws InterruptedException {
                    return now;
                }
            };
        }
    };

    private static final Logger LOGGER;
    private static final ThreadLocal<LocalChecker> localRateLimitChecker;
    private static final double MILLIS_PER_HOUR;
    private static Random ENTROPY;
    private static int EXPIRATION_WAIT_MILLIS;
    private static long NOTIFICATION_WAIT_MILLIS;
    private String displayName;

    static void setEntropy(Random random) {
        ENTROPY = random;
    }

    static void setExpirationWaitMillis(int expirationWaitMillis) {
        EXPIRATION_WAIT_MILLIS = expirationWaitMillis;
    }

    static void setNotificationWaitMillis(int notificationWaitMillis) {
        NOTIFICATION_WAIT_MILLIS = notificationWaitMillis;
    }

    private ApiRateLimitChecker(String displayName) {
        this.displayName = displayName;
    }

    public String getDisplayName() {
        return this.displayName;
    }

    public static void configureThreadLocalChecker(@NonNull TaskListener listener, @NonNull GitHub gitHub) {
        ApiRateLimitChecker.configureThreadLocalChecker(listener, gitHub.getApiUrl());
    }

    private static void configureThreadLocalChecker(TaskListener listener, String apiUrl) {
        LocalChecker checker = GitHubConfiguration.get().getApiRateLimitChecker().getChecker(listener, apiUrl);
        localRateLimitChecker.set(checker);
    }

    static void verifyConnection(GitHub gitHub) throws IOException {
        Objects.requireNonNull(gitHub);
        LocalChecker checker = ApiRateLimitChecker.getLocalChecker();
        try {
            TaskListener listener = checker != null ? checker.listener : new LogTaskListener(LOGGER, Level.INFO);
            localRateLimitChecker.set(NoThrottle.getChecker(listener, ""));
            gitHub.checkApiUrlValidity();
        }
        finally {
            localRateLimitChecker.set(checker);
        }
    }

    static LocalChecker getLocalChecker() {
        return localRateLimitChecker.get();
    }

    static void resetLocalChecker() {
        localRateLimitChecker.set(null);
    }

    @Deprecated
    public void checkApiRateLimit(TaskListener listener, GitHub gitHub) throws IOException, InterruptedException {
        ApiRateLimitChecker.configureThreadLocalChecker(listener, gitHub);
    }

    public abstract LocalChecker getChecker(@NonNull TaskListener var1, String var2);

    static int calculateBuffer(int limit) {
        return Math.max(15, limit / 20);
    }

    static int calculateNormalizedBurst(int rateLimit) {
        return rateLimit < 1000 ? Math.max(5, rateLimit / 10) : Math.max(200, rateLimit / 5);
    }

    static {
        LOGGER = Logger.getLogger(ApiRateLimitChecker.class.getName());
        localRateLimitChecker = new ThreadLocal();
        MILLIS_PER_HOUR = TimeUnit.HOURS.toMillis(1L);
        ENTROPY = new Random();
        EXPIRATION_WAIT_MILLIS = 65536;
        NOTIFICATION_WAIT_MILLIS = TimeUnit.MINUTES.toMillis(3L);
    }

    static abstract class LocalChecker {
        @NonNull
        private final TaskListener listener;
        private long expiration;

        LocalChecker(@NonNull TaskListener listener) {
            this.listener = Objects.requireNonNull(listener);
            this.resetExpiration();
        }

        protected boolean checkRateLimit(GHRateLimit.Record rateLimitRecord, long count) throws InterruptedException {
            long now;
            if (count == 0L) {
                this.resetExpiration();
            }
            if (this.waitUntilRateLimit(now = System.currentTimeMillis(), this.expiration, count)) {
                return true;
            }
            long newExpiration = this.checkRateLimitImpl(rateLimitRecord, count, now);
            if (newExpiration > this.expiration) {
                count = 0L;
            }
            return this.waitUntilRateLimit(now, newExpiration, count);
        }

        abstract long checkRateLimitImpl(@NonNull GHRateLimit.Record var1, long var2, long var4) throws InterruptedException;

        void resetExpiration() {
            this.expiration = Long.MIN_VALUE;
        }

        long calculateExpirationWhenBufferExceeded(GHRateLimit.Record rateLimit, long now, int buffer) {
            long expiration;
            long rateLimitResetMillis = rateLimit.getResetDate().getTime() - now;
            if (rateLimitResetMillis < 0L) {
                expiration = now + (long)ENTROPY.nextInt(EXPIRATION_WAIT_MILLIS);
                this.writeLog(String.format("Jenkins-Imposed API Limiter: Current quota for Github API usage has %d remaining (%d over budget). Next quota of %d due now. Sleeping for %s.", rateLimit.getRemaining(), buffer - rateLimit.getRemaining(), rateLimit.getLimit(), Util.getTimeSpanString((long)(1000L + expiration - now))));
            } else {
                expiration = rateLimit.getResetDate().getTime() + (long)ENTROPY.nextInt(EXPIRATION_WAIT_MILLIS);
                this.writeLog(String.format("Jenkins-Imposed API Limiter: Current quota for Github API usage has %d remaining (%d over budget). Next quota of %d in %s. Sleeping until reset.", rateLimit.getRemaining(), buffer - rateLimit.getRemaining(), rateLimit.getLimit(), Util.getTimeSpanString((long)rateLimitResetMillis)));
            }
            return expiration;
        }

        boolean waitUntilRateLimit(long now, long expiration, long count) throws InterruptedException {
            boolean waiting;
            boolean bl = waiting = expiration > now;
            if (waiting) {
                long nextNotify = now + NOTIFICATION_WAIT_MILLIS;
                this.expiration = expiration;
                if (count > 0L) {
                    this.writeLog(String.format("Jenkins-Imposed API Limiter: Still sleeping, now only %s remaining.", Util.getTimeSpanString((long)(expiration - now))));
                }
                if (Thread.interrupted()) {
                    throw new InterruptedException();
                }
                long sleep = Math.min(expiration, nextNotify) - now;
                if (sleep > 0L) {
                    Thread.sleep(sleep);
                }
            } else {
                this.resetExpiration();
            }
            return waiting;
        }

        void writeLog(String output) {
            this.listener.getLogger().println(GitHubConsoleNote.create(System.currentTimeMillis(), output));
        }
    }

    static final class RateLimitCheckerAdapter
    extends RateLimitChecker {
        RateLimitCheckerAdapter() {
        }

        protected boolean checkRateLimit(GHRateLimit.Record rateLimitRecord, long count) throws InterruptedException {
            LocalChecker checker = ApiRateLimitChecker.getLocalChecker();
            if (checker == null) {
                String apiUrl = "https://api.github.com";
                List<Endpoint> endpoints = GitHubConfiguration.get().getEndpoints();
                if (endpoints.size() > 0 && !StringUtils.isBlank((CharSequence)endpoints.get(0).getApiUri())) {
                    apiUrl = endpoints.get(0).getApiUri();
                }
                ApiRateLimitChecker.configureThreadLocalChecker((TaskListener)new LogTaskListener(LOGGER, Level.INFO), apiUrl);
                checker = ApiRateLimitChecker.getLocalChecker();
                checker.writeLog("LocalChecker for rate limit was not set for this thread. Configured using system settings with API URL '" + apiUrl + "'.");
            }
            return checker.checkRateLimit(rateLimitRecord, count);
        }
    }
}

