/*
 * Decompiled with CFR 0.152.
 */
package com.buildstash;

import com.buildstash.BuildstashUploadRequest;
import com.buildstash.BuildstashUploadResponse;
import com.buildstash.FileUploadInfo;
import com.buildstash.MultipartChunk;
import com.buildstash.PresignedData;
import com.buildstash.PresignedUrlResponse;
import com.buildstash.UploadRequestResponse;
import com.fasterxml.jackson.databind.ObjectMapper;
import hudson.FilePath;
import hudson.ProxyConfiguration;
import hudson.model.TaskListener;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.util.List;
import java.util.Map;

public class BuildstashUploadService {
    private static final String API_BASE_URL = "https://app.buildstash.com/api/v1";
    private static final String UPLOAD_REQUEST_ENDPOINT = "https://app.buildstash.com/api/v1/upload/request";
    private static final String UPLOAD_VERIFY_ENDPOINT = "https://app.buildstash.com/api/v1/upload/verify";
    private static final String MULTIPART_REQUEST_ENDPOINT = "https://app.buildstash.com/api/v1/upload/request/multipart";
    private static final String MULTIPART_EXPANSION_ENDPOINT = "https://app.buildstash.com/api/v1/upload/request/multipart/expansion";
    private final String apiKey;
    private final TaskListener listener;
    private final ObjectMapper objectMapper;
    private final HttpClient httpClient;

    public BuildstashUploadService(String apiKey, TaskListener listener) {
        this.apiKey = apiKey;
        this.listener = listener;
        this.objectMapper = new ObjectMapper();
        this.httpClient = ProxyConfiguration.newHttpClientBuilder().followRedirects(HttpClient.Redirect.NORMAL).build();
    }

    public BuildstashUploadResponse upload(BuildstashUploadRequest request) throws Exception {
        this.listener.getLogger().println("Requesting upload URLs from Buildstash...");
        UploadRequestResponse uploadRequestResponse = this.requestUploadUrls(request);
        this.listener.getLogger().println("Uploading files to Buildstash...");
        List<MultipartChunk> primaryFileParts = null;
        List<MultipartChunk> expansionFileParts = null;
        if (uploadRequestResponse.getPrimaryFile().isChunkedUpload()) {
            this.listener.getLogger().println("Uploading primary file using chunked upload...");
            primaryFileParts = this.uploadChunkedFile(request.getWorkspace().child(request.getPrimaryFilePath()), uploadRequestResponse.getPendingUploadId(), uploadRequestResponse.getPrimaryFile(), false);
        } else {
            this.listener.getLogger().println("Uploading primary file using direct upload...");
            this.uploadDirectFile(request.getWorkspace().child(request.getPrimaryFilePath()), uploadRequestResponse.getPrimaryFile().getPresignedData());
        }
        if (request.getExpansionFilePath() != null && uploadRequestResponse.getExpansionFiles() != null && !uploadRequestResponse.getExpansionFiles().isEmpty()) {
            FileUploadInfo expansionFile = uploadRequestResponse.getExpansionFiles().get(0);
            if (expansionFile.isChunkedUpload()) {
                this.listener.getLogger().println("Uploading expansion file using chunked upload...");
                expansionFileParts = this.uploadChunkedFile(request.getWorkspace().child(request.getExpansionFilePath()), uploadRequestResponse.getPendingUploadId(), expansionFile, true);
            } else {
                this.listener.getLogger().println("Uploading expansion file using direct upload...");
                this.uploadDirectFile(request.getWorkspace().child(request.getExpansionFilePath()), expansionFile.getPresignedData());
            }
        }
        this.listener.getLogger().println("Verifying upload...");
        return this.verifyUpload(uploadRequestResponse.getPendingUploadId(), primaryFileParts, expansionFileParts);
    }

    private UploadRequestResponse requestUploadUrls(BuildstashUploadRequest request) throws Exception {
        Map<String, Object> payload = request.toMap();
        String jsonPayload = this.objectMapper.writeValueAsString(payload);
        HttpRequest httpRequest = HttpRequest.newBuilder().uri(URI.create(UPLOAD_REQUEST_ENDPOINT)).header("Authorization", "Bearer " + this.apiKey).header("Content-Type", "application/json").POST(HttpRequest.BodyPublishers.ofString(jsonPayload)).build();
        HttpResponse<String> response = this.httpClient.send(httpRequest, HttpResponse.BodyHandlers.ofString());
        if (response.statusCode() != 200) {
            String responseBody = response.body();
            this.listener.error("Server returned error: " + response.statusCode());
            this.listener.error("Error response: " + responseBody);
            throw new RuntimeException("Failed to request upload URLs: " + response.statusCode() + " - " + responseBody);
        }
        String contentType = response.headers().firstValue("content-type").orElse("unknown");
        String responseBody = response.body();
        if (!contentType.contains("application/json") && !contentType.contains("json")) {
            throw new RuntimeException("Server returned HTML instead of JSON. This usually indicates an authentication error or the API endpoint is incorrect. Response content-type: " + contentType);
        }
        try {
            return (UploadRequestResponse)this.objectMapper.readValue(responseBody, UploadRequestResponse.class);
        }
        catch (Exception e) {
            throw new RuntimeException("Failed to parse JSON response: " + e.getMessage(), e);
        }
    }

    private List<MultipartChunk> uploadChunkedFile(FilePath filePath, String pendingUploadId, FileUploadInfo fileInfo, boolean isExpansion) throws Exception {
        String endpoint = isExpansion ? MULTIPART_EXPANSION_ENDPOINT : MULTIPART_REQUEST_ENDPOINT;
        long fileSize = filePath.length();
        int chunkSize = fileInfo.getChunkedPartSizeMb() * 1024 * 1024;
        int numberOfParts = fileInfo.getChunkedNumberParts();
        for (int i = 0; i < numberOfParts; ++i) {
            int partNumber = i + 1;
            long chunkStart = i * chunkSize;
            long chunkEnd = Math.min((long)((i + 1) * chunkSize - 1), fileSize - 1L);
            long contentLength = chunkEnd - chunkStart + 1L;
            this.listener.getLogger().println("Uploading chunked upload, part: " + partNumber + " of " + numberOfParts);
            PresignedUrlResponse presignedResponse = this.requestPresignedUrl(endpoint, pendingUploadId, partNumber, contentLength);
            this.uploadChunk(filePath, presignedResponse.getPartPresignedUrl(), chunkStart, chunkEnd, contentLength);
        }
        return null;
    }

    private PresignedUrlResponse requestPresignedUrl(String endpoint, String pendingUploadId, int partNumber, long contentLength) throws Exception {
        Map<String, Long> payload = Map.of("pending_upload_id", pendingUploadId, "part_number", partNumber, "content_length", contentLength);
        String jsonPayload = this.objectMapper.writeValueAsString(payload);
        HttpRequest httpRequest = HttpRequest.newBuilder().uri(URI.create(endpoint)).header("Authorization", "Bearer " + this.apiKey).header("Content-Type", "application/json").POST(HttpRequest.BodyPublishers.ofString(jsonPayload)).build();
        HttpResponse<String> response = this.httpClient.send(httpRequest, HttpResponse.BodyHandlers.ofString());
        if (response.statusCode() != 200) {
            throw new RuntimeException("Failed to get presigned URL: " + response.statusCode() + " - " + response.body());
        }
        return (PresignedUrlResponse)this.objectMapper.readValue(response.body(), PresignedUrlResponse.class);
    }

    private void uploadChunk(FilePath filePath, String presignedUrl, long start, long end, final long contentLength) throws Exception {
        try (InputStream inputStream = filePath.read();){
            long skipped = inputStream.skip(start);
            if (skipped != start) {
                throw new IOException("Failed to skip to position " + start + ", only skipped " + skipped + " bytes");
            }
            FilterInputStream limitedInputStream = new FilterInputStream(inputStream){
                private long remaining;
                {
                    super(in);
                    this.remaining = contentLength;
                }

                @Override
                public int read() throws IOException {
                    if (this.remaining <= 0L) {
                        return -1;
                    }
                    int result = super.read();
                    if (result != -1) {
                        --this.remaining;
                    }
                    return result;
                }

                @Override
                public int read(byte[] b, int off, int len) throws IOException {
                    if (this.remaining <= 0L) {
                        return -1;
                    }
                    int toRead = (int)Math.min((long)len, this.remaining);
                    int result = super.read(b, off, toRead);
                    if (result > 0) {
                        this.remaining -= (long)result;
                    }
                    return result;
                }
            };
            HttpRequest httpRequest = HttpRequest.newBuilder().uri(URI.create(presignedUrl)).header("Content-Type", "application/octet-stream").PUT(HttpRequest.BodyPublishers.ofInputStream(() -> limitedInputStream)).build();
            HttpResponse<String> response = this.httpClient.send(httpRequest, HttpResponse.BodyHandlers.ofString());
            if (response.statusCode() != 200) {
                throw new RuntimeException("Failed to upload chunk: " + response.statusCode() + " - " + response.body());
            }
        }
    }

    private void uploadDirectFile(FilePath filePath, PresignedData presignedData) throws Exception {
        byte[] fileBytes;
        String url = presignedData.getUrl();
        if (url == null || url.isBlank()) {
            throw new RuntimeException("Presigned URL is null or empty");
        }
        String contentType = presignedData.getHeaderAsString("Content-Type");
        String contentDisposition = presignedData.getHeaderAsString("Content-Disposition");
        String xAmzAcl = presignedData.getHeaderAsString("x-amz-acl");
        long fileSize = filePath.length();
        try (InputStream inputStream = filePath.read();){
            fileBytes = inputStream.readAllBytes();
            if ((long)fileBytes.length != fileSize) {
                throw new RuntimeException(String.format("File read mismatch: expected %d bytes, but read %d bytes", fileSize, fileBytes.length));
            }
        }
        HttpRequest.Builder requestBuilder = HttpRequest.newBuilder().uri(URI.create(url));
        if (contentType != null) {
            requestBuilder.header("Content-Type", contentType);
        }
        if (contentDisposition != null) {
            requestBuilder.header("Content-Disposition", contentDisposition);
        }
        if (xAmzAcl != null) {
            requestBuilder.header("x-amz-acl", xAmzAcl);
        }
        requestBuilder.PUT(HttpRequest.BodyPublishers.ofByteArray(fileBytes));
        HttpRequest httpRequest = requestBuilder.build();
        HttpResponse<String> response = this.httpClient.send(httpRequest, HttpResponse.BodyHandlers.ofString());
        if (response.statusCode() != 200) {
            throw new RuntimeException("Failed to upload file: " + response.statusCode() + " - " + response.body());
        }
    }

    private BuildstashUploadResponse verifyUpload(String pendingUploadId, List<MultipartChunk> primaryFileParts, List<MultipartChunk> expansionFileParts) throws Exception {
        Map<String, String> payload = Map.of("pending_upload_id", pendingUploadId);
        String jsonPayload = this.objectMapper.writeValueAsString(payload);
        HttpRequest httpRequest = HttpRequest.newBuilder().uri(URI.create(UPLOAD_VERIFY_ENDPOINT)).header("Authorization", "Bearer " + this.apiKey).header("Content-Type", "application/json").POST(HttpRequest.BodyPublishers.ofString(jsonPayload)).build();
        HttpResponse<String> response = this.httpClient.send(httpRequest, HttpResponse.BodyHandlers.ofString());
        if (response.statusCode() != 200) {
            throw new RuntimeException("Failed to verify upload: " + response.statusCode() + " - " + response.body());
        }
        return (BuildstashUploadResponse)this.objectMapper.readValue(response.body(), BuildstashUploadResponse.class);
    }

    public void close() throws IOException {
    }
}

