/*
 * Decompiled with CFR 0.152.
 */
package jenkins.bouncycastle.api;

import edu.umd.cs.findbugs.annotations.CheckForNull;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;
import java.io.File;
import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.Writer;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.security.Key;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.Provider;
import java.security.PublicKey;
import java.security.Security;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.interfaces.RSAPrivateCrtKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.RSAPublicKeySpec;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HexFormat;
import java.util.List;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.apache.commons.io.FileUtils;
import org.bouncycastle.asn1.pkcs.PrivateKeyInfo;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import org.bouncycastle.openssl.PEMDecryptorProvider;
import org.bouncycastle.openssl.PEMEncryptedKeyPair;
import org.bouncycastle.openssl.PEMKeyPair;
import org.bouncycastle.openssl.PEMParser;
import org.bouncycastle.openssl.jcajce.JcaPEMKeyConverter;
import org.bouncycastle.openssl.jcajce.JcaPEMWriter;
import org.bouncycastle.openssl.jcajce.JcePEMDecryptorProviderBuilder;
import org.bouncycastle.operator.InputDecryptorProvider;
import org.bouncycastle.pkcs.PKCS8EncryptedPrivateKeyInfo;
import org.bouncycastle.pkcs.PKCSException;
import org.bouncycastle.pkcs.jcajce.JcePKCSPBEInputDecryptorProviderBuilder;

public final class PEMEncodable {
    @NonNull
    private final Object object;
    private static final HexFormat HEX_FORMAT = HexFormat.ofDelimiter(":");
    private static final Provider BOUNCY_CASTLE_PROVIDER;
    private static final Logger LOGGER;

    private PEMEncodable(@NonNull Object pemObject) {
        this.object = pemObject;
    }

    @NonNull
    public static PEMEncodable create(@NonNull Key key) {
        return new PEMEncodable(key);
    }

    @NonNull
    public static PEMEncodable create(@NonNull KeyPair keyPair) {
        return new PEMEncodable(keyPair);
    }

    @NonNull
    public static PEMEncodable create(@NonNull Certificate certificate) {
        return new PEMEncodable(certificate);
    }

    @NonNull
    public static PEMEncodable decode(@NonNull String pem) throws IOException, UnrecoverableKeyException {
        return PEMEncodable.decode(pem, null);
    }

    @NonNull
    public static PEMEncodable decode(@NonNull String pem, @Nullable char[] passphrase) throws IOException, UnrecoverableKeyException {
        List<PEMEncodable> pems = PEMEncodable.decodeAll(pem, passphrase);
        if (pems.isEmpty()) {
            throw new IOException("Could not parse PEM, only key pairs, private keys, public keys and certificates are supported");
        }
        if (pems.size() > 1) {
            throw new IOException("Expected a single PEM entry, but got " + pems.size());
        }
        return pems.get(0);
    }

    @NonNull
    public static List<PEMEncodable> decodeAll(@NonNull String pem, @Nullable char[] passphrase) throws IOException, UnrecoverableKeyException {
        ArrayList<PEMEncodable> objects = new ArrayList<PEMEncodable>();
        try (PEMParser parser = new PEMParser((Reader)new StringReader(pem));){
            Object object = parser.readObject();
            while (object != null) {
                objects.add(PEMEncodable.convertedPemToPemDecodable(object, passphrase));
                object = parser.readObject();
            }
        }
        return objects;
    }

    private static final PEMEncodable convertedPemToPemDecodable(Object object, char[] passphrase) throws UnrecoverableKeyException, IOException {
        try {
            JcaPEMKeyConverter kConv = new JcaPEMKeyConverter().setProvider(BOUNCY_CASTLE_PROVIDER);
            if (object instanceof PEMEncryptedKeyPair) {
                if (passphrase != null) {
                    PEMDecryptorProvider dp = new JcePEMDecryptorProviderBuilder().build(passphrase);
                    PEMEncryptedKeyPair ekp = (PEMEncryptedKeyPair)object;
                    return new PEMEncodable(kConv.getKeyPair(ekp.decryptKeyPair(dp)));
                }
                throw new UnrecoverableKeyException("Key is passphrase protected, but no passphrase was provided");
            }
            if (object instanceof PKCS8EncryptedPrivateKeyInfo) {
                if (passphrase != null) {
                    InputDecryptorProvider dp = new JcePKCSPBEInputDecryptorProviderBuilder().setProvider(BOUNCY_CASTLE_PROVIDER).build(passphrase);
                    PKCS8EncryptedPrivateKeyInfo epk = (PKCS8EncryptedPrivateKeyInfo)object;
                    PrivateKey pk = kConv.getPrivateKey(epk.decryptPrivateKeyInfo(dp));
                    return PEMEncodable.getPEMEncodableKeyPairFromPrivateKey(pk);
                }
                throw new UnrecoverableKeyException("Key is passphrase protected, but no passphrase was provided");
            }
            if (object instanceof PEMKeyPair) {
                return new PEMEncodable(kConv.getKeyPair((PEMKeyPair)object));
            }
            if (object instanceof PrivateKeyInfo) {
                PrivateKey pk = kConv.getPrivateKey((PrivateKeyInfo)object);
                return PEMEncodable.getPEMEncodableKeyPairFromPrivateKey(pk);
            }
            if (object instanceof SubjectPublicKeyInfo) {
                return new PEMEncodable(kConv.getPublicKey((SubjectPublicKeyInfo)object));
            }
            if (object instanceof X509CertificateHolder) {
                JcaX509CertificateConverter cConv = new JcaX509CertificateConverter().setProvider(BOUNCY_CASTLE_PROVIDER);
                return new PEMEncodable(cConv.getCertificate((X509CertificateHolder)object));
            }
            throw new IOException("Could not parse PEM, only key pairs, private keys, public keys and certificates are supported. Received " + object.getClass().getName());
        }
        catch (InvalidKeySpecException | PKCSException e) {
            UnrecoverableKeyException unrecoverableKeyEx = new UnrecoverableKeyException(e.getMessage());
            unrecoverableKeyEx.initCause(e);
            throw unrecoverableKeyEx;
        }
        catch (CertificateException e) {
            throw new IOException("Could not read certificate", e);
        }
        catch (NoSuchAlgorithmException e) {
            throw new IOException("Algorithm required for parsing is not implemented", e);
        }
        catch (AssertionError e) {
            if (e.getClass().getName().equals("org.bouncycastle.crypto.fips.FipsUnapprovedOperationError")) {
                UnrecoverableKeyException unrecoverableKeyEx = new UnrecoverableKeyException("Provided Object is not FIPS 140 compliant");
                unrecoverableKeyEx.initCause((Throwable)((Object)e));
                throw unrecoverableKeyEx;
            }
            throw e;
        }
    }

    private static PEMEncodable getPEMEncodableKeyPairFromPrivateKey(PrivateKey pk) throws NoSuchAlgorithmException, InvalidKeySpecException {
        if (pk instanceof RSAPrivateCrtKey) {
            RSAPrivateCrtKey rsaPK = (RSAPrivateCrtKey)pk;
            RSAPublicKeySpec pubKeySpec = new RSAPublicKeySpec(rsaPK.getModulus(), rsaPK.getPublicExponent());
            KeyFactory kf = KeyFactory.getInstance("RSA");
            return new PEMEncodable(new KeyPair(kf.generatePublic(pubKeySpec), rsaPK));
        }
        return new PEMEncodable(pk);
    }

    @NonNull
    public String encode() throws IOException {
        StringWriter sw = new StringWriter();
        try (JcaPEMWriter w = new JcaPEMWriter((Writer)sw);){
            w.writeObject(this.object);
        }
        return sw.toString();
    }

    @NonNull
    public static PEMEncodable read(@NonNull File pemFile) throws IOException, UnrecoverableKeyException {
        return PEMEncodable.read(pemFile, null);
    }

    @NonNull
    public static PEMEncodable read(@NonNull File pemFile, @Nullable char[] passphrase) throws IOException, UnrecoverableKeyException {
        return PEMEncodable.decode(FileUtils.readFileToString((File)pemFile, (Charset)StandardCharsets.UTF_8), passphrase);
    }

    @NonNull
    public static List<PEMEncodable> readAll(@NonNull File pemFile) throws IOException, UnrecoverableKeyException {
        return PEMEncodable.readAll(pemFile, null);
    }

    @NonNull
    public static List<PEMEncodable> readAll(@NonNull File pemFile, @Nullable char[] passphrase) throws IOException, UnrecoverableKeyException {
        return PEMEncodable.decodeAll(FileUtils.readFileToString((File)pemFile, (Charset)StandardCharsets.UTF_8), passphrase);
    }

    public void write(@NonNull File pemFile) throws IOException {
        FileUtils.writeStringToFile((File)pemFile, (String)this.encode(), (Charset)StandardCharsets.UTF_8);
    }

    @CheckForNull
    public KeyPair toKeyPair() {
        if (this.object instanceof KeyPair) {
            return (KeyPair)this.object;
        }
        return null;
    }

    @CheckForNull
    public PublicKey toPublicKey() {
        if (this.object instanceof PublicKey) {
            return (PublicKey)this.object;
        }
        if (this.object instanceof KeyPair) {
            return ((KeyPair)this.object).getPublic();
        }
        if (this.object instanceof Certificate) {
            return ((Certificate)this.object).getPublicKey();
        }
        return null;
    }

    @CheckForNull
    public Certificate toCertificate() {
        if (this.object instanceof Certificate) {
            return (Certificate)this.object;
        }
        return null;
    }

    @CheckForNull
    public PrivateKey toPrivateKey() {
        if (this.object instanceof PrivateKey) {
            return (PrivateKey)this.object;
        }
        if (this.object instanceof KeyPair) {
            return ((KeyPair)this.object).getPrivate();
        }
        return null;
    }

    @CheckForNull
    public Object getRawObject() {
        return this.object;
    }

    @CheckForNull
    public String getPrivateKeyFingerprint() {
        PrivateKey key = this.toPrivateKey();
        if (key == null) {
            return null;
        }
        return HEX_FORMAT.formatHex(PEMEncodable.getKeyDigestSHA1(key));
    }

    @CheckForNull
    public String getPublicKeyFingerprint() {
        PublicKey key = this.toPublicKey();
        if (key == null) {
            return null;
        }
        return HEX_FORMAT.formatHex(PEMEncodable.getKeyDigestMD5(key));
    }

    @NonNull
    public static byte[] getKeyDigestSHA1(@NonNull Key k) {
        try {
            return PEMEncodable.getKeyDigest(k, "SHA1");
        }
        catch (NoSuchAlgorithmException e) {
            throw new AssertionError((Object)"SHA1 algorithm support is mandated by Java Language Specification. See https://docs.oracle.com/javase/7/docs/api/java/security/MessageDigest.html");
        }
    }

    @NonNull
    public static byte[] getKeyDigestMD5(@NonNull Key k) {
        try {
            return PEMEncodable.getKeyDigest(k, "MD5");
        }
        catch (NoSuchAlgorithmException e) {
            throw new AssertionError((Object)"MD5 algorithm support is mandated by Java Language Specification. See https://docs.oracle.com/javase/7/docs/api/java/security/MessageDigest.html");
        }
    }

    @NonNull
    public static byte[] getKeyDigest(@NonNull Key k, @NonNull String algorithm) throws NoSuchAlgorithmException {
        MessageDigest md = MessageDigest.getInstance(algorithm);
        md.update(k.getEncoded());
        return md.digest();
    }

    static {
        Provider provider = Security.getProvider("BCFIPS");
        if (provider == null) {
            provider = Security.getProvider("BC");
        }
        if (provider == null) {
            throw new IllegalStateException(Arrays.asList(Security.getProviders()).stream().map(p -> p.getName()).collect(Collectors.joining(",", "Couldn't locate either of bouncy castle FIPS or non fips provider, available providers are", ".")));
        }
        BOUNCY_CASTLE_PROVIDER = provider;
        LOGGER = Logger.getLogger(PEMEncodable.class.getName());
    }
}

