/*
 * Decompiled with CFR 0.152.
 */
package io.jenkins.plugins.security.scan.service.bridge;

import hudson.FilePath;
import hudson.model.TaskListener;
import io.jenkins.plugins.security.scan.bridge.BridgeDownloadParameters;
import io.jenkins.plugins.security.scan.exception.PluginExceptionHandler;
import io.jenkins.plugins.security.scan.global.LoggerWrapper;
import io.jenkins.plugins.security.scan.global.Utility;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Map;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class BridgeDownloadParametersService {
    private final TaskListener listener;
    private final LoggerWrapper logger;
    private final FilePath workspace;

    public BridgeDownloadParametersService(FilePath workspace, TaskListener listener) {
        this.workspace = workspace;
        this.listener = listener;
        this.logger = new LoggerWrapper(listener);
    }

    public boolean performBridgeDownloadParameterValidation(BridgeDownloadParameters bridgeDownloadParameters) throws PluginExceptionHandler {
        boolean validUrl = this.isValidUrl(bridgeDownloadParameters.getBridgeDownloadUrl());
        boolean validVersion = this.isValidVersion(bridgeDownloadParameters.getBridgeDownloadVersion());
        boolean validInstallationPath = this.isValidInstallationPath(bridgeDownloadParameters.getBridgeInstallationPath());
        if (validUrl && validVersion && validInstallationPath) {
            this.logger.info("Bridge download parameters are validated successfully", new Object[0]);
            return true;
        }
        this.logger.error("Bridge download parameters are not valid", new Object[0]);
        throw new PluginExceptionHandler(105);
    }

    public boolean isValidUrl(String url) {
        if (url.isEmpty()) {
            this.logger.warn("Provided Bridge CLI URL cannot be empty", new Object[0]);
            return false;
        }
        try {
            new URL(url);
            return true;
        }
        catch (MalformedURLException me) {
            this.logger.error("Provided Bridge CLI URL is not valid: %s", me.getMessage());
            return false;
        }
    }

    public boolean isValidVersion(String version) {
        Pattern pattern = Pattern.compile("[0-9.]+[a-zA-Z0-9]*");
        Matcher matcher = pattern.matcher(version);
        if (matcher.matches() || version.equals("latest")) {
            return true;
        }
        this.logger.error("Provided Bridge CLI version not found in artifactory: %s", version);
        return false;
    }

    public boolean isValidInstallationPath(String installationPath) {
        try {
            FilePath path = new FilePath(this.workspace.getChannel(), installationPath);
            FilePath parentPath = path.getParent();
            if (parentPath != null && parentPath.exists() && parentPath.isDirectory()) {
                FilePath tempFile = parentPath.createTempFile("temp", null);
                boolean isWritable = tempFile.delete();
                if (isWritable) {
                    return true;
                }
                this.logger.error("The bridge installation parent path: %s is not writable", parentPath.toURI());
                return false;
            }
            if (parentPath == null || !parentPath.exists()) {
                this.logger.error("The bridge installation parent path: %s doesn't exist", path.toURI().toString());
            } else if (!parentPath.isDirectory()) {
                this.logger.error("The bridge installation parent path: %s is not a directory", parentPath.toURI().toString());
            }
            return false;
        }
        catch (IOException | InterruptedException e) {
            this.logger.error("An exception occurred while validating the installation path: %s", e.getMessage());
            Thread.currentThread().interrupt();
            return false;
        }
    }

    public BridgeDownloadParameters getBridgeDownloadParams(Map<String, Object> scanParameters, BridgeDownloadParameters bridgeDownloadParameters) {
        boolean isNetworkAirgap;
        if (scanParameters.containsKey("bridgecli_install_directory")) {
            bridgeDownloadParameters.setBridgeInstallationPath(scanParameters.get("bridgecli_install_directory").toString().trim());
        }
        boolean bl = isNetworkAirgap = scanParameters.containsKey("network_airgap") && scanParameters.get("network_airgap").equals(true);
        if (scanParameters.containsKey("bridgecli_download_url")) {
            bridgeDownloadParameters.setBridgeDownloadUrl(scanParameters.get("bridgecli_download_url").toString().trim());
            String extractedVersionNumber = Utility.extractVersionFromUrl(bridgeDownloadParameters.getBridgeDownloadUrl());
            if (!extractedVersionNumber.equals("NA")) {
                bridgeDownloadParameters.setBridgeDownloadVersion(extractedVersionNumber);
            }
        } else if (scanParameters.containsKey("bridgecli_download_version") && !isNetworkAirgap) {
            String desiredVersion = scanParameters.get("bridgecli_download_version").toString().trim();
            String bridgeDownloadUrl = String.join((CharSequence)"/", "https://repo.blackduck.com/bds-integrations-release/com/blackduck/integration/bridge/binaries/bridge-cli-bundle", desiredVersion, this.getBridgeZipFileName(desiredVersion));
            bridgeDownloadParameters.setBridgeDownloadUrl(bridgeDownloadUrl);
            bridgeDownloadParameters.setBridgeDownloadVersion(desiredVersion);
        } else if (!isNetworkAirgap) {
            String bridgeDownloadUrl = String.join((CharSequence)"/", "https://repo.blackduck.com/bds-integrations-release/com/blackduck/integration/bridge/binaries/bridge-cli-bundle", "latest", this.getBridgeZipFileName());
            bridgeDownloadParameters.setBridgeDownloadUrl(bridgeDownloadUrl);
        }
        return bridgeDownloadParameters;
    }

    public void updateBridgeInstallationPath(BridgeDownloadParameters bridgeDownloadParameters) {
        String separator = Utility.getDirectorySeparator(this.workspace, this.listener);
        String modifiedInstalationPath = bridgeDownloadParameters.getBridgeInstallationPath().concat(separator).concat("bridge-cli-bundle").concat("-").concat(this.getPlatform(bridgeDownloadParameters.getBridgeDownloadVersion()));
        bridgeDownloadParameters.setBridgeInstallationPath(modifiedInstalationPath);
    }

    public String getPlatform(String version) {
        boolean isArm;
        String os = Utility.getAgentOs(this.workspace, this.listener);
        if (os.contains("win")) {
            return "win64";
        }
        String arch = Utility.getAgentOsArch(this.workspace, this.listener);
        boolean isMac = os.contains("mac");
        boolean isLinux = os.contains("linux");
        boolean bl = isArm = arch.startsWith("arm") || arch.startsWith("aarch");
        if (isMac) {
            boolean isValidVersionForARM = Objects.isNull(version) || Utility.isVersionCompatible(version, "2.1.0");
            return this.selectPlatform(version, isArm, isValidVersionForARM, "macos_arm", "macosx", "2.1.0");
        }
        if (isLinux) {
            boolean isValidVersionForARM = Objects.isNull(version) || Utility.isVersionCompatible(version, "3.5.1");
            return this.selectPlatform(version, isArm, isValidVersionForARM, "linux_arm", "linux64", "3.5.1");
        }
        return "linux64";
    }

    private String selectPlatform(String version, boolean isArm, boolean isValidVersionForARM, String armPlatform, String defaultPlatform, String minVersion) {
        if (isArm && !isValidVersionForARM) {
            this.logger.info(String.format("Detected Bridge CLI version (%s) below the minimum ARM support requirement (%s). Defaulting to %s platform.", version, minVersion, defaultPlatform), new Object[0]);
            return defaultPlatform;
        }
        return isArm && isValidVersionForARM ? armPlatform : defaultPlatform;
    }

    public String getBridgeZipFileName() {
        return "bridge-cli-bundle".concat("-").concat(this.getPlatform(null)).concat(".zip");
    }

    public String getBridgeZipFileName(String version) {
        return "bridge-cli-bundle".concat("-").concat(version).concat("-").concat(this.getPlatform(version)).concat(".zip");
    }
}

