/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.bitbucket.jenkins.internal.http;

import com.atlassian.bitbucket.jenkins.internal.client.HttpRequestExecutor;
import com.atlassian.bitbucket.jenkins.internal.client.RequestConfiguration;
import com.atlassian.bitbucket.jenkins.internal.client.exception.AuthorizationException;
import com.atlassian.bitbucket.jenkins.internal.client.exception.BadRequestException;
import com.atlassian.bitbucket.jenkins.internal.client.exception.BitbucketClientException;
import com.atlassian.bitbucket.jenkins.internal.client.exception.ConnectionFailureException;
import com.atlassian.bitbucket.jenkins.internal.client.exception.NotFoundException;
import com.atlassian.bitbucket.jenkins.internal.client.exception.RateLimitedException;
import com.atlassian.bitbucket.jenkins.internal.client.exception.ServerErrorException;
import com.atlassian.bitbucket.jenkins.internal.client.exception.UnhandledErrorException;
import com.atlassian.bitbucket.jenkins.internal.http.RetryOnRateLimitConfig;
import com.atlassian.bitbucket.jenkins.internal.util.SystemPropertyUtils;
import hudson.Plugin;
import java.io.IOException;
import java.io.InputStream;
import java.net.ConnectException;
import java.net.SocketTimeoutException;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.Nullable;
import javax.inject.Inject;
import jenkins.model.Jenkins;
import okhttp3.Call;
import okhttp3.Headers;
import okhttp3.HttpUrl;
import okhttp3.Interceptor;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import okhttp3.ResponseBody;

public class HttpRequestExecutorImpl
implements HttpRequestExecutor {
    private static final int BAD_REQUEST_FAMILY = 4;
    private static final int HTTP_RATE_LIMITED = 429;
    private static final MediaType JSON = MediaType.parse((String)"application/json; charset=utf-8");
    private static final int SERVER_ERROR_FAMILY = 5;
    private static final Logger log = Logger.getLogger(HttpRequestExecutorImpl.class.getName());
    private final Call.Factory httpCallFactory;

    @Inject
    public HttpRequestExecutorImpl() {
        this((Call.Factory)HttpRequestExecutorImpl.buildDefaultOkHttpClient());
    }

    public HttpRequestExecutorImpl(Call.Factory httpCallFactory) {
        this.httpCallFactory = httpCallFactory;
    }

    public static OkHttpClient buildDefaultOkHttpClient() {
        long connectionTimeout = SystemPropertyUtils.parsePositiveLongFromSystemProperty("bitbucket.http.connect.timeout", 30000L);
        long readTimeout = SystemPropertyUtils.parsePositiveLongFromSystemProperty("bitbucket.http.request.timeout", 30000L);
        log.info("buildDefaultHttpCallFactory with: bitbucket.http.connect.timeout: " + connectionTimeout + ", bitbucket.http.request.timeout: " + readTimeout);
        return new OkHttpClient.Builder().connectTimeout(connectionTimeout, TimeUnit.MILLISECONDS).readTimeout(readTimeout, TimeUnit.MILLISECONDS).addInterceptor((Interceptor)new UserAgentInterceptor()).build();
    }

    @Override
    public void executeDelete(HttpUrl url, RequestConfiguration ... additionalConfig) {
        Request.Builder requestBuilder = new Request.Builder().url(url).delete();
        this.executeRequest(requestBuilder, HttpRequestExecutor.ResponseConsumer.EMPTY_RESPONSE, additionalConfig);
    }

    @Override
    public <T> T executeGet(HttpUrl url, HttpRequestExecutor.ResponseConsumer<T> consumer, RequestConfiguration ... additionalConfig) {
        Request.Builder requestBuilder = new Request.Builder().url(url);
        return this.executeRequest(requestBuilder, consumer, additionalConfig);
    }

    @Override
    public <T> T executePost(HttpUrl url, String requestBodyAsJson, HttpRequestExecutor.ResponseConsumer<T> consumer, RequestConfiguration ... additionalConfig) {
        Request.Builder requestBuilder = new Request.Builder().post(RequestBody.create((MediaType)JSON, (String)requestBodyAsJson)).url(url);
        return this.executeRequest(requestBuilder, consumer, additionalConfig);
    }

    @Override
    public InputStream executeStreamingGet(HttpUrl url, HttpRequestExecutor.ResponseConsumer<InputStream> consumer, RequestConfiguration ... additionalConfig) {
        return this.performStreamingGet(url, additionalConfig);
    }

    @Override
    public <T> T executePut(HttpUrl url, String requestBodyAsJson, HttpRequestExecutor.ResponseConsumer<T> consumer, RequestConfiguration ... additionalConfig) {
        Request.Builder requestBuilder = new Request.Builder().put(RequestBody.create((MediaType)JSON, (String)requestBodyAsJson)).url(url);
        return this.executeRequest(requestBuilder, consumer, additionalConfig);
    }

    private <T> T executeRequest(Request.Builder requestBuilder, HttpRequestExecutor.ResponseConsumer<T> consumer, RequestConfiguration ... additionalConfigs) {
        Set<RequestConfiguration> additionalConfig = this.toSet(additionalConfigs);
        additionalConfig.forEach(config -> config.apply(requestBuilder));
        return this.performRequest(requestBuilder.build(), consumer);
    }

    private Response makeRequest(Request request) {
        try {
            Response response = this.httpCallFactory.newCall(request).execute();
            int responseCode = response.code();
            ResponseBody body = response.body();
            if (response.isSuccessful()) {
                log.fine("Bitbucket - call successful");
                return response;
            }
            this.handleError(responseCode, body == null ? null : body.string(), response.headers());
        }
        catch (ConnectException | SocketTimeoutException e) {
            log.log(Level.FINE, "Bitbucket - Connection failed", e);
            throw new ConnectionFailureException(e);
        }
        catch (IOException e) {
            log.log(Level.FINE, "Bitbucket - io exception", e);
            throw new BitbucketClientException(e);
        }
        catch (RateLimitedException e) {
            RetryOnRateLimitConfig rateLimitConfig = (RetryOnRateLimitConfig)request.tag(RetryOnRateLimitConfig.class);
            if (rateLimitConfig != null && rateLimitConfig.incrementAndGetAttempts() <= rateLimitConfig.getMaxAttempts()) {
                try {
                    Thread.sleep(e.getRetryIn());
                }
                catch (InterruptedException ex) {
                    throw new UnhandledErrorException("Interrupted during wait to retry", -2, null);
                }
                return this.makeRequest(request);
            }
            throw e;
        }
        throw new UnhandledErrorException("Unhandled error", -1, null);
    }

    private void handleError(int responseCode, @Nullable String body, Headers headers) throws AuthorizationException {
        switch (responseCode) {
            case 401: 
            case 403: {
                log.info("Bitbucket - responded with not authorized ");
                throw new AuthorizationException("Provided credentials cannot access the resource", responseCode, body);
            }
            case 404: {
                log.info("Bitbucket - Path not found");
                throw new NotFoundException("The requested resource does not exist", body);
            }
            case 429: {
                throw new RateLimitedException("Rate limited", 429, body, headers);
            }
        }
        int family = responseCode / 100;
        switch (family) {
            case 4: {
                log.info("Bitbucket - did not accept the request");
                throw new BadRequestException("The request is malformed", responseCode, body);
            }
            case 5: {
                log.info("Bitbucket - failed to service request");
                throw new ServerErrorException("The server failed to service request", responseCode, body);
            }
        }
        throw new UnhandledErrorException("Unhandled error", responseCode, body);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private <T> T performRequest(Request request, HttpRequestExecutor.ResponseConsumer<T> consumer) {
        Response response = null;
        try {
            response = this.makeRequest(request);
            T t = consumer.consume(response);
            return t;
        }
        finally {
            if (response != null && response.body() != null) {
                response.close();
            }
        }
    }

    private InputStream performStreamingGet(HttpUrl url, RequestConfiguration ... additionalConfigs) {
        Set<RequestConfiguration> additionalConfig = this.toSet(additionalConfigs);
        Request.Builder requestBuilder = new Request.Builder().url(url);
        additionalConfig.forEach(config -> config.apply(requestBuilder));
        Response response = this.makeRequest(requestBuilder.build());
        return response.body().byteStream();
    }

    private Set<RequestConfiguration> toSet(RequestConfiguration[] additionalConfig) {
        HashSet<RequestConfiguration> configurations = new HashSet<RequestConfiguration>(additionalConfig.length, 1.0f);
        for (RequestConfiguration config : additionalConfig) {
            if (configurations.add(config)) continue;
            throw new IllegalArgumentException("Duplicate RequestConfiguration provided");
        }
        return configurations;
    }

    private static class UserAgentInterceptor
    implements Interceptor {
        private final String bbJenkinsUserAgent;

        UserAgentInterceptor() {
            String version = "unknown";
            try {
                Plugin plugin = Jenkins.get().getPlugin("atlassian-bitbucket-server-integration");
                if (plugin != null) {
                    version = plugin.getWrapper().getVersion();
                }
            }
            catch (IllegalStateException e) {
                Logger.getLogger(UserAgentInterceptor.class.getName()).log(Level.WARNING, "Jenkins not available", e);
            }
            this.bbJenkinsUserAgent = "bitbucket-jenkins-integration/" + version;
        }

        public Response intercept(Interceptor.Chain chain) throws IOException {
            Request request = chain.request().newBuilder().header("User-Agent", this.bbJenkinsUserAgent).build();
            return chain.proceed(request);
        }
    }
}

