/*
 * Decompiled with CFR 0.152.
 */
package net.sourceforge.pmd.util;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.Spliterators;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import net.sourceforge.pmd.util.AssertionUtil;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.checkerframework.checker.nullness.qual.Nullable;

public final class IteratorUtil {
    private static final int MATCH_ANY = 0;
    private static final int MATCH_ALL = 1;
    private static final int MATCH_NONE = 2;

    private IteratorUtil() {
    }

    public static <T> Iterator<T> takeWhile(final Iterator<T> iter, final Predicate<? super T> predicate) {
        return new AbstractIterator<T>(){

            @Override
            protected void computeNext() {
                Object next;
                if (iter.hasNext() && predicate.test(next = iter.next())) {
                    this.setNext(next);
                    return;
                }
                this.done();
            }
        };
    }

    public static <T> Iterator<T> reverse(Iterator<T> it) {
        List<T> tmp = IteratorUtil.toList(it);
        Collections.reverse(tmp);
        return tmp.iterator();
    }

    public static <T, R> Iterator<R> flatMap(final Iterator<? extends T> iter, final Function<? super T, ? extends @Nullable Iterator<? extends R>> f) {
        return new AbstractIterator<R>(){
            private Iterator<? extends R> current = null;

            @Override
            protected void computeNext() {
                if (this.current != null && this.current.hasNext()) {
                    this.setNext(this.current.next());
                } else {
                    while (iter.hasNext()) {
                        Iterator next = (Iterator)f.apply(iter.next());
                        if (next == null || !next.hasNext()) continue;
                        this.current = next;
                        this.setNext(this.current.next());
                        return;
                    }
                    this.done();
                }
            }
        };
    }

    public static <R> Iterator<R> flatMapWithSelf(final Iterator<? extends R> iter, final Function<? super R, ? extends @Nullable Iterator<? extends R>> f) {
        return new AbstractIterator<R>(){
            private Iterator<? extends R> current = null;

            @Override
            protected void computeNext() {
                if (this.current != null && this.current.hasNext()) {
                    this.setNext(this.current.next());
                } else {
                    this.current = null;
                    if (iter.hasNext()) {
                        Object next = iter.next();
                        this.setNext(next);
                        this.current = (Iterator)f.apply(next);
                    } else {
                        this.done();
                    }
                }
            }
        };
    }

    public static <T> Iterator<@NonNull T> filterNotNull(Iterator<? extends T> it) {
        return IteratorUtil.filter(it, Objects::nonNull);
    }

    public static <T, R> Iterator<@NonNull R> mapNotNull(final Iterator<? extends T> it, final Function<@NonNull ? super T, @Nullable ? extends R> mapper) {
        return new AbstractIterator<R>(){

            @Override
            protected void computeNext() {
                while (it.hasNext()) {
                    Object map;
                    Object next = it.next();
                    if (next == null || (map = mapper.apply(next)) == null) continue;
                    this.setNext(map);
                    return;
                }
                this.done();
            }
        };
    }

    public static <T> Iterator<T> filter(final Iterator<? extends T> it, final Predicate<? super T> filter) {
        return new AbstractIterator<T>(){

            @Override
            protected void computeNext() {
                while (it.hasNext()) {
                    Object next = it.next();
                    if (!filter.test(next)) continue;
                    this.setNext(next);
                    return;
                }
                this.done();
            }
        };
    }

    public static <T> Iterator<T> peek(Iterator<? extends T> iter, Consumer<? super T> action) {
        return IteratorUtil.map(iter, it -> {
            action.accept(it);
            return it;
        });
    }

    public static <T, R> Iterator<R> map(final Iterator<? extends T> iter, final Function<? super T, ? extends R> mapper) {
        return new Iterator<R>(){

            @Override
            public boolean hasNext() {
                return iter.hasNext();
            }

            @Override
            public R next() {
                return mapper.apply(iter.next());
            }
        };
    }

    public static <T, R> Iterable<R> mapIterator(Iterable<? extends T> iter, Function<? super Iterator<? extends T>, ? extends Iterator<R>> mapper) {
        return () -> (Iterator)mapper.apply(iter.iterator());
    }

    @SafeVarargs
    public static <T> Iterator<T> iterate(T ... elements) {
        return Arrays.asList(elements).iterator();
    }

    public static <T> Iterator<T> concat(final Iterator<? extends T> as, final Iterator<? extends T> bs) {
        if (!as.hasNext()) {
            return bs;
        }
        if (!bs.hasNext()) {
            return as;
        }
        return new Iterator<T>(){

            @Override
            public boolean hasNext() {
                return as.hasNext() || bs.hasNext();
            }

            @Override
            public T next() {
                return as.hasNext() ? as.next() : bs.next();
            }
        };
    }

    public static <T> Iterator<T> distinct(Iterator<? extends T> iter) {
        HashSet seen = new HashSet();
        return IteratorUtil.filter(iter, seen::add);
    }

    public static <T> List<T> toList(Iterator<? extends T> it) {
        ArrayList<T> list = new ArrayList<T>();
        while (it.hasNext()) {
            list.add(it.next());
        }
        return list;
    }

    public static <T> List<@NonNull T> toNonNullList(Iterator<? extends @Nullable T> it) {
        ArrayList<@NonNull T> list = new ArrayList<T>();
        while (it.hasNext()) {
            T next = it.next();
            if (next == null) continue;
            list.add(next);
        }
        return list;
    }

    public static <T> Iterator<@NonNull T> dropLast(final Iterator<? extends @Nullable T> it, int n) {
        AssertionUtil.requireNonNegative("n", n);
        if (n == 0) {
            return IteratorUtil.coerceWildcard(it);
        }
        if (n == 1) {
            if (!it.hasNext()) {
                return Collections.emptyIterator();
            }
            return new AbstractIterator<T>(){
                T next;
                {
                    this.next = it.next();
                }

                @Override
                protected void computeNext() {
                    if (it.hasNext()) {
                        this.setNext(this.next);
                        this.next = it.next();
                    } else {
                        this.done();
                    }
                }
            };
        }
        final Object[] ringBuffer = new Object[n];
        for (int i = 0; i < n && it.hasNext(); ++i) {
            ringBuffer[i] = it.next();
        }
        if (!it.hasNext()) {
            return Collections.emptyIterator();
        }
        return new AbstractIterator<T>(){
            private int idx = 0;

            @Override
            protected void computeNext() {
                if (it.hasNext()) {
                    this.setNext(ringBuffer[this.idx]);
                    ringBuffer[this.idx] = it.next();
                    this.idx = (this.idx + 1) % ringBuffer.length;
                } else {
                    this.done();
                }
            }
        };
    }

    public static <T> Iterator<T> coerceWildcard(Iterator<? extends T> it) {
        return it;
    }

    public static <T> Iterable<T> toIterable(Iterator<T> it) {
        return () -> it;
    }

    public static int count(Iterator<?> it) {
        int count = 0;
        while (it.hasNext()) {
            it.next();
            ++count;
        }
        return count;
    }

    public static <T> @Nullable T last(Iterator<? extends T> iterator) {
        T next = null;
        while (iterator.hasNext()) {
            next = iterator.next();
        }
        return next;
    }

    public static <T> @Nullable T getNth(Iterator<? extends T> iterator, int n) {
        IteratorUtil.advance(iterator, n);
        return iterator.hasNext() ? (T)iterator.next() : null;
    }

    public static void advance(Iterator<?> iterator, int n) {
        AssertionUtil.requireNonNegative("n", n);
        while (n > 0 && iterator.hasNext()) {
            iterator.next();
            --n;
        }
    }

    public static <T> Iterator<T> take(final Iterator<? extends T> iterator, final int n) {
        AssertionUtil.requireNonNegative("n", n);
        if (n == 0) {
            return Collections.emptyIterator();
        }
        return new AbstractIterator<T>(){
            private int yielded = 0;

            @Override
            protected void computeNext() {
                if (this.yielded >= n || !iterator.hasNext()) {
                    this.done();
                } else {
                    this.setNext(iterator.next());
                }
                ++this.yielded;
            }
        };
    }

    public static <T> Iterator<T> drop(final Iterator<? extends T> source, final int n) {
        AssertionUtil.requireNonNegative("n", n);
        if (n == 0) {
            return source;
        }
        return new AbstractIterator<T>(){
            private int yielded = 0;

            @Override
            protected void computeNext() {
                while (this.yielded++ < n && source.hasNext()) {
                    source.next();
                }
                if (!source.hasNext()) {
                    this.done();
                } else {
                    this.setNext(source.next());
                }
            }
        };
    }

    public static <T> Iterator<@NonNull T> generate(final @Nullable T seed, final Function<? super @NonNull T, ? extends @Nullable T> stepper) {
        return new AbstractIterator<T>(){
            T next;
            {
                this.next = seed;
            }

            @Override
            protected void computeNext() {
                if (this.next == null) {
                    this.done();
                    return;
                }
                this.setNext(this.next);
                this.next = stepper.apply(this.next);
            }
        };
    }

    public static <T> boolean anyMatch(Iterator<? extends T> iterator, Predicate<? super T> pred) {
        return IteratorUtil.matches(iterator, pred, 0);
    }

    public static <T> boolean allMatch(Iterator<? extends T> iterator, Predicate<? super T> pred) {
        return IteratorUtil.matches(iterator, pred, 1);
    }

    public static <T> boolean noneMatch(Iterator<? extends T> iterator, Predicate<? super T> pred) {
        return IteratorUtil.matches(iterator, pred, 2);
    }

    private static <T> boolean matches(Iterator<? extends T> iterator, Predicate<? super T> pred, int matchKind) {
        boolean kindAll;
        boolean kindAny = matchKind == 0;
        boolean bl = kindAll = matchKind == 1;
        while (iterator.hasNext()) {
            T value = iterator.next();
            boolean match = pred.test(value);
            if (!(match ^ kindAll)) continue;
            return kindAny && match;
        }
        return !kindAny;
    }

    public static <T> Iterator<T> singletonIterator(T value) {
        class SingletonIterator
        implements Iterator<T> {
            private boolean done;
            final /* synthetic */ Object val$value;

            SingletonIterator(Object object) {
                this.val$value = object;
            }

            @Override
            public boolean hasNext() {
                return !this.done;
            }

            @Override
            public T next() {
                if (this.done) {
                    throw new NoSuchElementException();
                }
                this.done = true;
                return this.val$value;
            }

            @Override
            public void forEachRemaining(Consumer<? super T> action) {
                action.accept(this.val$value);
            }
        }
        return new SingletonIterator(value);
    }

    public static <T> Iterable<T> asReversed(final List<T> lst) {
        return () -> new Iterator<T>(){
            ListIterator<T> li;
            {
                this.li = lst.listIterator(lst.size());
            }

            @Override
            public boolean hasNext() {
                return this.li.hasPrevious();
            }

            @Override
            public T next() {
                return this.li.previous();
            }

            @Override
            public void remove() {
                this.li.remove();
            }
        };
    }

    public static <T> Stream<T> toStream(Iterator<? extends T> iter) {
        return StreamSupport.stream(Spliterators.spliteratorUnknownSize(iter, 0), false);
    }

    public static <T> Stream<T> toStream(Iterable<T> iter) {
        if (iter instanceof Collection) {
            return ((Collection)iter).stream();
        }
        return StreamSupport.stream(iter.spliterator(), false);
    }

    public static abstract class AbstractPausingIterator<T>
    extends AbstractIterator<T> {
        private int numYielded = 0;
        private T currentValue;

        @Override
        public T next() {
            Object next = super.next();
            this.currentValue = next;
            this.prepareViewOn(next);
            ++this.numYielded;
            return next;
        }

        protected void prepareViewOn(T current) {
        }

        protected final int getIterationCount() {
            return this.numYielded;
        }

        protected T getCurrentValue() {
            this.ensureReadable();
            return this.currentValue;
        }

        protected void ensureReadable() {
            if (this.numYielded == 0) {
                throw new IllegalStateException("No values were yielded, should have called next");
            }
        }
    }

    public static abstract class AbstractIterator<T>
    implements Iterator<T> {
        private State state = State.NOT_READY;
        private T next = null;

        @Override
        public boolean hasNext() {
            switch (this.state) {
                case DONE: {
                    return false;
                }
                case READY: {
                    return true;
                }
            }
            this.state = null;
            this.computeNext();
            if (this.state == null) {
                throw new IllegalStateException("Should have called done or setNext");
            }
            return this.state == State.READY;
        }

        @Override
        public T next() {
            if (!this.hasNext()) {
                throw new NoSuchElementException();
            }
            this.state = State.NOT_READY;
            return this.next;
        }

        protected final void setNext(T t) {
            assert (this.state == null) : "Must call exactly one of setNext or done";
            this.next = t;
            this.state = State.READY;
        }

        protected final void done() {
            assert (this.state == null) : "Must call exactly one of setNext or done";
            this.state = State.DONE;
        }

        protected abstract void computeNext();

        @Override
        public final void remove() {
            throw new UnsupportedOperationException("remove");
        }

        static enum State {
            READY,
            NOT_READY,
            DONE;

        }
    }
}

