/*
 * Decompiled with CFR 0.152.
 */
package net.sourceforge.pmd.lang.document;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.Adler32;
import java.util.zip.CheckedInputStream;
import java.util.zip.Checksum;
import net.sourceforge.pmd.lang.document.Chars;
import net.sourceforge.pmd.lang.document.SourceCodePositioner;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class TextFileContent {
    private static final Logger LOGGER = LoggerFactory.getLogger(TextFileContent.class);
    private static final String CRLF = "\r\n";
    private static final String LF = "\n";
    private static final String CR = "\r";
    public static final String NORMALIZED_LINE_TERM = "\n";
    public static final char NORMALIZED_LINE_TERM_CHAR = '\n';
    private static final int DEFAULT_BUFSIZE = 8192;
    private static final Pattern NEWLINE_PATTERN = Pattern.compile("\r\n?|\n");
    private static final String FALLBACK_LINESEP = System.lineSeparator();
    private final Chars cdata;
    private final String lineTerminator;
    private final long checkSum;
    private final SourceCodePositioner positioner;

    private TextFileContent(Chars normalizedText, String lineTerminator, long checkSum, SourceCodePositioner positioner) {
        this.cdata = normalizedText;
        this.lineTerminator = lineTerminator;
        this.checkSum = checkSum;
        this.positioner = positioner;
    }

    public Chars getNormalizedText() {
        return this.cdata;
    }

    public String getLineTerminator() {
        return this.lineTerminator;
    }

    public long getCheckSum() {
        return this.checkSum;
    }

    SourceCodePositioner getPositioner() {
        return this.positioner;
    }

    public static @NonNull TextFileContent fromCharSeq(CharSequence text) {
        return TextFileContent.normalizeCharSeq(text, FALLBACK_LINESEP);
    }

    public static TextFileContent fromReader(Reader reader) throws IOException {
        try (Reader r = reader;){
            TextFileContent textFileContent = TextFileContent.normalizingRead(r, 8192, FALLBACK_LINESEP, TextFileContent.newChecksum(), true);
            return textFileContent;
        }
    }

    public static TextFileContent fromInputStream(InputStream inputStream, Charset sourceEncoding) throws IOException {
        return TextFileContent.fromInputStream(inputStream, sourceEncoding, FALLBACK_LINESEP);
    }

    static TextFileContent fromInputStream(InputStream inputStream, Charset sourceEncoding, String fallbackLineSep) throws IOException {
        Checksum checksum = TextFileContent.newChecksum();
        try (CheckedInputStream checkedIs = new CheckedInputStream(new BufferedInputStream(inputStream), checksum);){
            TextFileContent textFileContent;
            try (InputStreamReader reader = new InputStreamReader((InputStream)checkedIs, sourceEncoding);){
                textFileContent = TextFileContent.normalizingRead(reader, 8192, fallbackLineSep, checksum, false);
            }
            return textFileContent;
        }
    }

    static @NonNull TextFileContent normalizeCharSeq(CharSequence text, String fallBackLineSep) {
        long checksum = TextFileContent.getCheckSum(text);
        if (text.length() > 0 && text.charAt(0) == '\ufeff') {
            text = text.subSequence(1, text.length());
        }
        Matcher matcher = NEWLINE_PATTERN.matcher(text);
        boolean needsNormalization = false;
        String lineTerminator = null;
        while (matcher.find()) {
            if ("\n".equals(lineTerminator = TextFileContent.detectLineTerm(lineTerminator, matcher.group(), fallBackLineSep))) continue;
            needsNormalization = true;
            if (!lineTerminator.equals(fallBackLineSep)) continue;
        }
        if (lineTerminator == null) {
            lineTerminator = fallBackLineSep;
            needsNormalization = false;
        }
        if (needsNormalization) {
            text = NEWLINE_PATTERN.matcher(text).replaceAll("\n");
        }
        return new TextFileContent(Chars.wrap(text), lineTerminator, checksum, SourceCodePositioner.create(text));
    }

    static TextFileContent normalizingRead(Reader input, int bufSize, String fallbackLineSep) throws IOException {
        return TextFileContent.normalizingRead(input, bufSize, fallbackLineSep, TextFileContent.newChecksum(), true);
    }

    static TextFileContent normalizingRead(Reader input, int bufSize, String fallbackLineSep, Checksum checksum, boolean updateChecksum) throws IOException {
        char[] cbuf = new char[bufSize];
        StringBuilder result = new StringBuilder(bufSize);
        String detectedLineTerm = null;
        boolean afterCr = false;
        SourceCodePositioner.Builder positionerBuilder = new SourceCodePositioner.Builder();
        int bufOffset = 0;
        int nextCharToCopy = 0;
        int n = input.read(cbuf);
        if (n > 0 && cbuf[0] == '\ufeff') {
            nextCharToCopy = 1;
        }
        while (n != -1) {
            if (updateChecksum) {
                TextFileContent.updateChecksum(checksum, CharBuffer.wrap(cbuf, nextCharToCopy, n));
            }
            int offsetDiff = 0;
            for (int i = nextCharToCopy; i < n; ++i) {
                char c = cbuf[i];
                if (afterCr || c == '\n') {
                    int newLineOffset;
                    String newLineTerm;
                    if (afterCr && c != '\n') {
                        newLineTerm = CR;
                        newLineOffset = bufOffset + i + offsetDiff;
                        if (i > 0) {
                            cbuf[i - 1] = 10;
                        } else {
                            result.append("\n");
                        }
                    } else {
                        if (afterCr) {
                            newLineTerm = CRLF;
                            if (i > 0) {
                                cbuf[i - 1] = 10;
                                result.append(cbuf, nextCharToCopy, i - nextCharToCopy);
                                nextCharToCopy = i + 1;
                            }
                            --offsetDiff;
                        } else {
                            newLineTerm = "\n";
                        }
                        newLineOffset = bufOffset + i + offsetDiff + 1;
                    }
                    positionerBuilder.addLineEndAtOffset(newLineOffset);
                    detectedLineTerm = TextFileContent.detectLineTerm(detectedLineTerm, newLineTerm, fallbackLineSep);
                }
                afterCr = c == '\r';
            }
            if (nextCharToCopy != n) {
                int numRemaining = n - nextCharToCopy;
                if (afterCr) {
                    --numRemaining;
                }
                result.append(cbuf, nextCharToCopy, numRemaining);
            }
            nextCharToCopy = 0;
            bufOffset += n + offsetDiff;
            n = input.read(cbuf);
        }
        if (afterCr) {
            result.append("\n");
            positionerBuilder.addLineEndAtOffset(bufOffset);
            detectedLineTerm = TextFileContent.detectLineTerm(detectedLineTerm, CR, fallbackLineSep);
        }
        if (detectedLineTerm == null) {
            detectedLineTerm = fallbackLineSep;
        }
        return new TextFileContent(Chars.wrap(result), detectedLineTerm, checksum.getValue(), positionerBuilder.build(bufOffset));
    }

    private static String detectLineTerm(@Nullable String curLineTerm, String newLineTerm, String fallback) {
        if (curLineTerm == null) {
            return newLineTerm;
        }
        if (curLineTerm.equals(newLineTerm)) {
            return curLineTerm;
        }
        LOGGER.debug("Detect mixed line terminators. Falling back to system default.");
        return fallback;
    }

    private static long getCheckSum(CharSequence cs) {
        Checksum checksum = TextFileContent.newChecksum();
        TextFileContent.updateChecksum(checksum, CharBuffer.wrap(cs));
        return checksum.getValue();
    }

    private static void updateChecksum(Checksum checksum, CharBuffer chars) {
        ByteBuffer bytes = StandardCharsets.UTF_8.encode(chars);
        assert (bytes.hasArray()) : "Encoder should produce a heap buffer";
        checksum.update(bytes.array(), bytes.arrayOffset(), bytes.remaining());
    }

    private static Checksum newChecksum() {
        return new Adler32();
    }
}

